<?php
require_once('source/dbconfig.php');
require_once('source/function.php');
session_start();

// Verify exam session
if (!isset($_SESSION['current_exam']) || !isset($_GET['exam_subject_id'])) {
    header("Location: welcome.php");
    exit();
}

// Load exam questions
try {
    // First get the subject_id from exam_subject table
    $stmt = $DBH->prepare("
        SELECT subject_id, exam_id 
        FROM tbl_exam_subject 
        WHERE id = :exam_subject_id
        LIMIT 1
    ");
    $stmt->execute(['exam_subject_id' => $_GET['exam_subject_id']]);
    $examSubject = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$examSubject) {
        throw new Exception("Invalid exam subject selected");
    }
    
    // Now get the exam_class_id from exam_class table
    $stmt = $DBH->prepare("
        SELECT id 
        FROM tbl_exam_class 
        WHERE exam_id = :exam_id
        AND class = :class
        AND term = :term
        AND session = :session
        LIMIT 1
    ");
    $stmt->execute([
        'exam_id' => $examSubject['exam_id'],
        'class' => $_SESSION['current_exam']['class'],
        'term' => $_SESSION['exam_info']['term'],
        'session' => $_SESSION['exam_info']['session']
    ]);
    $examClass = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$examClass) {
        throw new Exception("Exam not configured for your class");
    }
    
    // Now get the questions
    $stmt = $DBH->prepare("
        SELECT 
            q.id,
            q.question_text,
            q.option_a,
            q.option_b,
            q.option_c,
            q.option_d,
            q.correct_answer,
            q.file_path
        FROM tbl_question q
        WHERE q.subject_id = :subject_id
        AND q.exam_class_id = :exam_class_id
        ORDER BY RAND()
        LIMIT 1000
    ");
    
    $stmt->execute([
        'subject_id' => $examSubject['subject_id'],
        'exam_class_id' => $examClass['id']
    ]);
    
    $questions = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    if (empty($questions)) {
        throw new Exception("No questions available for this exam. Please contact your instructor.");
    }
    
    // Update session with correct IDs
    $_SESSION['current_exam']['subject_id'] = $examSubject['subject_id'];
    $_SESSION['current_exam']['exam_class_id'] = $examClass['id'];
    $_SESSION['current_exam']['question_ids'] = array_column($questions, 'id');
    
} catch (Exception $e) {
    die("Error: " . $e->getMessage());
}

// Calculate end time
$endTime = $_SESSION['current_exam']['start_time'] + ($_SESSION['current_exam']['duration'] * 60);
$timeLeft = $endTime - time();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Exam - <?= htmlspecialchars($_SESSION['current_exam']['subject_name']) ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        :root {
            --light-green: #e8f5e9;
            --primary-green: #2e7d32;
            --secondary-green: #4caf50;
            --accent-green: #8bc34a;
            --dark-green: #1b5e20;
        }
        
        body {
            background-color: var(--light-green);
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        
        .question-container {
            max-width: 800px;
            margin: 0 auto;
            padding-bottom: 100px;
            background-color: white;
            border-radius: 15px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.05);
            padding: 30px;
            margin-top: 20px;
        }
        
        .timer-container {
            position: sticky;
            top: 0;
            background: white;
            z-index: 1000;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            padding: 10px 0;
            border-bottom: 3px solid var(--primary-green);
        }
        
        .timer-danger {
            color: #dc3545;
            animation: pulse 1s infinite;
        }
        
        @keyframes pulse {
            0% { opacity: 1; }
            50% { opacity: 0.5; }
            100% { opacity: 1; }
        }
        
        .question-card {
            margin-bottom: 25px;
            border-left: 5px solid var(--primary-green);
            transition: all 0.3s ease;
            display: none;
            border-radius: 10px;
            overflow: hidden;
        }
        
        .question-card.active {
            display: block;
            animation: fadeIn 0.5s ease;
            box-shadow: 0 3px 10px rgba(0,0,0,0.08);
        }
        
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(10px); }
            to { opacity: 1; transform: translateY(0); }
        }
        
        .option-label {
            display: block;
            padding: 15px 20px;
            margin: 10px 0;
            border: 2px solid #dee2e6;
            border-radius: 10px;
            cursor: pointer;
            transition: all 0.2s;
            font-size: 1.05rem;
        }
        
        .option-label:hover {
            background-color: #f1f8ff;
            border-color: var(--accent-green);
        }
        
        .option-input:checked + .option-label {
            background-color: #e8f5e9;
            border-color: var(--secondary-green);
            box-shadow: 0 0 0 3px rgba(76, 175, 80, 0.2);
        }
        
        .question-media {
            margin-bottom: 25px;
            text-align: center;
            background-color: #f8f9fa;
            padding: 15px;
            border-radius: 10px;
        }
        
        .question-media img, .question-media video {
            max-width: 100%;
            max-height: 300px;
            border-radius: 8px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        
        .navigation-btns {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            background: white;
            padding: 15px;
            box-shadow: 0 -2px 15px rgba(0,0,0,0.1);
            z-index: 1000;
            display: flex;
            justify-content: space-between;
            align-items: center;
            border-top: 3px solid var(--primary-green);
        }
        
        .progress-indicator {
            font-size: 1rem;
            color: var(--dark-green);
            margin: 0 15px;
            font-weight: 500;
        }
        
        .btn-navigation {
            min-width: 120px;
            font-weight: 500;
            padding: 8px 20px;
        }
        
        .btn-primary {
            background-color: var(--primary-green);
            border-color: var(--primary-green);
        }
        
        .btn-primary:hover {
            background-color: var(--dark-green);
            border-color: var(--dark-green);
        }
        
        .btn-success {
            background-color: var(--secondary-green);
            border-color: var(--secondary-green);
        }
        
        .btn-success:hover {
            background-color: var(--dark-green);
            border-color: var(--dark-green);
        }
        
        .badge.bg-primary {
            background-color: var(--primary-green) !important;
            font-size: 0.9rem;
            padding: 8px 12px;
        }
        
        .card-header {
            background-color: #f8f9fa;
            border-bottom: 2px solid var(--accent-green);
            font-weight: 600;
        }
        
        /* Fullscreen warning styles */
        #fullscreen-warning {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0,0,0,0.9);
            color: white;
            display: flex;
            flex-direction: column;
            justify-content: center;
            align-items: center;
            z-index: 9999;
            font-size: 1.5rem;
            text-align: center;
            padding: 20px;
        }
        
        #enter-fullscreen {
            background-color: var(--secondary-green);
            border-color: var(--secondary-green);
            padding: 10px 25px;
            font-size: 1.2rem;
            margin-top: 20px;
        }
        
        /* Enhanced submit confirmation modal */
        #submitConfirmModal .modal-content {
            border-radius: 15px;
            overflow: hidden;
            border: none;
            box-shadow: 0 5px 20px rgba(0,0,0,0.2);
        }
        
        #submitConfirmModal .modal-header {
            background-color: var(--light-green);
            border-bottom: 3px solid var(--primary-green);
            padding: 1.5rem;
            text-align: center;
        }
        
        #submitConfirmModal .modal-title {
            color: var(--dark-green);
            font-weight: 700;
            font-size: 1.5rem;
        }
        
        #submitConfirmModal .modal-body {
            padding: 2rem;
            text-align: center;
        }
        
        #submitConfirmModal .alert-warning {
            background-color: #fff8e1;
            border-left: 4px solid #ffc107;
            text-align: left;
        }
        
        #submitConfirmModal .modal-footer {
            border-top: none;
            justify-content: center;
            padding: 1.5rem;
            background-color: #f8f9fa;
        }
        
        #submitConfirmModal .btn-outline-secondary {
            border-width: 2px;
            font-weight: 500;
        }
        
        #submitConfirmModal .btn-success {
            background-color: var(--secondary-green);
            border-color: var(--secondary-green);
            font-weight: 500;
        }
    </style>
</head>
<body>
    <!-- Fullscreen warning (initially hidden) -->
    <div id="fullscreen-warning" style="display: none;">
        <h2>Exam Requirement</h2>
        <p>You must be in fullscreen mode to continue with the exam.</p>
        <button id="enter-fullscreen" class="btn btn-primary btn-lg">Enter Fullscreen</button>
    </div>

    <!-- Submit confirmation modal -->
<!-- Replace the existing modal with this enhanced version -->
<div class="modal fade" id="submitConfirmModal" tabindex="-1" aria-hidden="true" data-bs-backdrop="static">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content border-0 shadow-lg">
            <div class="modal-header border-0 bg-warning bg-opacity-10">
                <div class="text-center w-100">
                    <i class="bi bi-exclamation-triangle-fill text-warning fs-1 mb-3"></i>
                    <h3 class="modal-title fw-bold text-dark">Final Submission Confirmation</h3>
                </div>
            </div>
            <div class="modal-body text-center py-4">
                <h5 class="fw-bold mb-3">You're about to submit your exam!</h5>
                <div class="alert alert-warning border-warning border-2">
                    <i class="bi bi-check-circle-fill me-2"></i>
                    <strong>Please verify:</strong>
                    <ul class="text-start mt-2 ps-3">
                        <li>All questions have been answered</li>
                        <li>You've reviewed your answers</li>
                        <li>Time remaining: <span id="remaining-time-display" class="fw-bold"><?= gmdate("H:i:s", $timeLeft) ?></span></li>
                    </ul>
                </div>
                <p class="lead">Are you absolutely sure you want to submit now?</p>
            </div>
            <div class="modal-footer border-0 justify-content-center">
                <button type="button" class="btn btn-lg btn-outline-danger px-4 mx-2" data-bs-dismiss="modal">
                    <i class="bi bi-arrow-left-circle me-2"></i>No, Continue Exam
                </button>
                <button type="button" class="btn btn-lg btn-success px-4 mx-2" id="confirmSubmit">
                    <i class="bi bi-check-circle me-2"></i>Yes, Submit Now
                </button>
            </div>
        </div>
    </div>
</div>

    <div class="timer-container">
        <div class="container text-center">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <h4 class="mb-0">
                        Time Remaining: 
                        <span id="timer" class="<?= $timeLeft < 300 ? 'timer-danger' : '' ?>">
                            <?= gmdate("H:i:s", $timeLeft) ?>
                        </span>
                    </h4>
                    <div class="progress mt-2" style="height: 8px;">
                        <div id="time-progress" class="progress-bar bg-primary" 
                             role="progressbar" 
                             style="width: 100%" 
                             aria-valuenow="100" 
                             aria-valuemin="0" 
                             aria-valuemax="100"></div>
                    </div>
                </div>
                <div>
                    <span class="badge bg-primary">
                        <?= htmlspecialchars($_SESSION['current_exam']['subject_name']) ?>
                    </span>
                </div>
            </div>
        </div>
    </div>

    <div class="container py-4 question-container">
        <form id="exam-form" action="submit_exam.php" method="post">
            <input type="hidden" name="exam_subject_id" value="<?= $_SESSION['current_exam']['exam_subject_id'] ?>">
            <input type="hidden" name="subject" value="<?= htmlspecialchars($_SESSION['current_exam']['subject_name']) ?>">
            <input type="hidden" name="class" value="<?= htmlspecialchars($_SESSION['current_exam']['class']) ?>">
            <input type="hidden" name="term" value="<?= htmlspecialchars($_SESSION['exam_info']['term']) ?>">
            <input type="hidden" name="session" value="<?= htmlspecialchars($_SESSION['exam_info']['session']) ?>">
            
            <?php foreach ($questions as $index => $question): ?>
                <div class="question-card card mb-4" id="question-<?= $index ?>" <?= $index === 0 ? 'class="active"' : '' ?>>
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h5 class="mb-0">Question <?= $index + 1 ?></h5>
                        <span class="badge bg-secondary"><?= count($questions) ?> total</span>
                    </div>
                    <div class="card-body">
                        <?php if (!empty($question['file_path'])): ?>
                            <div class="question-media">
                                <?php if (preg_match('/\.(jpg|jpeg|png|gif|webp)$/i', $question['file_path'])): ?>
                                    <img src="<?= htmlspecialchars($question['file_path']) ?>" class="img-fluid" alt="Question Image">
                                <?php elseif (preg_match('/\.(mp4|webm|ogg)$/i', $question['file_path'])): ?>
                                    <video controls class="img-fluid">
                                        <source src="<?= htmlspecialchars($question['file_path']) ?>">
                                        Your browser does not support the video tag.
                                    </video>
                                <?php endif; ?>
                            </div>
                        <?php endif; ?>
                        
                        <p class="card-text fs-5"><?= htmlspecialchars($question['question_text']) ?></p>
                        
                        <div class="options mt-4">
                            <?php foreach (['A' => 'option_a', 'B' => 'option_b', 'C' => 'option_c', 'D' => 'option_d'] as $letter => $option): ?>
                                <div class="form-check mb-3">
                                    <input class="form-check-input option-input" 
                                           type="radio" 
                                           name="answers[<?= $question['id'] ?>]" 
                                           id="q<?= $question['id'] ?>-<?= $letter ?>" 
                                           value="<?= $letter ?>">
                                    <label class="form-check-label option-label" for="q<?= $question['id'] ?>-<?= $letter ?>">
                                        <strong class="me-2"><?= $letter ?>.</strong> 
                                        <span><?= htmlspecialchars($question[$option]) ?></span>
                                    </label>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
            <?php endforeach; ?>
            
            <div class="navigation-btns">
                <button type="button" class="btn btn-secondary btn-navigation" id="prev-btn" disabled>
                    <i class="bi bi-chevron-left"></i> Previous
                </button>
                
                <div class="progress-indicator">
                    Question <span id="current-q">1</span> of <?= count($questions) ?>
                </div>
                
                <button type="button" class="btn btn-primary btn-navigation" id="next-btn">
                    Next <i class="bi bi-chevron-right"></i>
                </button>
                
                <button type="button" class="btn btn-success" id="submit-btn" style="display:none;">
                    <i class="bi bi-check-circle"></i> Submit Exam
                </button>
            </div>
        </form>
    </div>

    <!-- Bootstrap Icons -->
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.8.0/font/bootstrap-icons.css">
    <!-- Bootstrap JS Bundle with Popper -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Fullscreen enforcement
            const fullscreenWarning = document.getElementById('fullscreen-warning');
            const enterFullscreenBtn = document.getElementById('enter-fullscreen');
            
            function checkFullscreen() {
                if (!document.fullscreenElement) {
                    fullscreenWarning.style.display = 'flex';
                    return false;
                }
                fullscreenWarning.style.display = 'none';
                return true;
            }
            
            function requestFullscreen() {
                if (!document.fullscreenElement) {
                    document.documentElement.requestFullscreen().catch(err => {
                        console.error(`Error attempting to enable fullscreen: ${err.message}`);
                    });
                }
            }
            
            enterFullscreenBtn.addEventListener('click', function() {
                requestFullscreen();
            });
            
            document.addEventListener('fullscreenchange', checkFullscreen);
            document.addEventListener('webkitfullscreenchange', checkFullscreen);
            document.addEventListener('mozfullscreenchange', checkFullscreen);
            document.addEventListener('MSFullscreenChange', checkFullscreen);
            
            // Initial check
            if (!checkFullscreen()) {
                requestFullscreen();
            }
            
            // Timer functionality
            const endTime = <?= $endTime ?>;
            const totalDuration = <?= $_SESSION['current_exam']['duration'] * 60 ?>;
            const timerElement = document.getElementById('timer');
            const progressBar = document.getElementById('time-progress');
            
            function updateTimer() {
                const now = Math.floor(Date.now() / 1000);
                const remaining = endTime - now;
                
                if (remaining <= 0) {
                    timerElement.textContent = "00:00:00";
                    progressBar.style.width = "0%";
                    // Show confirmation modal when time expires
                    const submitModal = new bootstrap.Modal(document.getElementById('submitConfirmModal'));
                    submitModal.show();
                    return;
                }
                
                const hours = Math.floor(remaining / 3600);
                const minutes = Math.floor((remaining % 3600) / 60);
                const seconds = remaining % 60;
                
                timerElement.textContent = 
                    `${hours.toString().padStart(2, '0')}:${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
                
                const progressPercent = (remaining / totalDuration) * 100;
                progressBar.style.width = `${progressPercent}%`;
                
                if (remaining < 300) {
                    timerElement.classList.add('timer-danger');
                }
            }
            
            const timerInterval = setInterval(updateTimer, 1000);
            updateTimer();

            // Question navigation
            const questions = Array.from(document.querySelectorAll('.question-card'));
            const totalQuestions = questions.length;
            let currentQuestion = 0;
            
            const prevBtn = document.getElementById('prev-btn');
            const nextBtn = document.getElementById('next-btn');
            const submitBtn = document.getElementById('submit-btn');
            const currentQIndicator = document.getElementById('current-q');
            
            // Initialize navigation
            function showQuestion(index) {
                // Hide all questions
                questions.forEach(question => {
                    question.classList.remove('active');
                });
                
                // Show current question with animation
                questions[index].classList.add('active');
                
                // Update button states
                updateButtonStates();
            }
            
            function updateButtonStates() {
                prevBtn.disabled = currentQuestion === 0;
                nextBtn.style.display = currentQuestion < totalQuestions - 1 ? 'block' : 'none';
                submitBtn.style.display = currentQuestion >= totalQuestions - 1 ? 'block' : 'none';
                currentQIndicator.textContent = currentQuestion + 1;
            }
            
            // Navigation functions
            function goToPreviousQuestion() {
                if (currentQuestion > 0) {
                    currentQuestion--;
                    showQuestion(currentQuestion);
                }
            }
            
            function goToNextQuestion() {
                if (currentQuestion < totalQuestions - 1) {
                    currentQuestion++;
                    showQuestion(currentQuestion);
                }
            }
            
            // Event listeners
            prevBtn.addEventListener('click', goToPreviousQuestion);
            nextBtn.addEventListener('click', goToNextQuestion);
            
            // Keyboard navigation
            document.addEventListener('keydown', function(e) {
                if (e.key === 'ArrowLeft') {
                    goToPreviousQuestion();
                } else if (e.key === 'ArrowRight') {
                    goToNextQuestion();
                }
            });
            
            // Auto-save answers
            document.querySelectorAll('.option-input').forEach(input => {
                input.addEventListener('change', function() {
                    const formData = new FormData(document.getElementById('exam-form'));
                    
                    fetch('save_answer.php', {
                        method: 'POST',
                        body: formData
                    }).catch(error => console.error('Error saving answer:', error));
                });
            });
            
            // Submit confirmation handling
            const examForm = document.getElementById('exam-form');
            const submitModal = new bootstrap.Modal(document.getElementById('submitConfirmModal'));
            
            submitBtn.addEventListener('click', function(e) {
                e.preventDefault();
                submitModal.show();
            });
            
            document.getElementById('confirmSubmit').addEventListener('click', function() {
                submitModal.hide();
                examForm.submit();
            });
            
            // Initialize first question 
            showQuestion(currentQuestion);
        });
    </script>
</body>
</html>