<?php
require_once('source/session.php');
require_once('source/error.php');
require_once('source/dbconfig.php');
require_once('source/function.php');
require_once('source/restrict_admin.php');
session_start();    

// Check if the user is logged in
if (isset($_SESSION['SESS_ID_ADMIN'])) {
    $userId = $_SESSION['SESS_ID_ADMIN'];

    // Fetch user details based on the ID
    $query = "SELECT * FROM tbl_admin WHERE id = :id";
    $stmt = $DBH->prepare($query);
    $stmt->bindParam(':id', $userId, PDO::PARAM_INT);
    $stmt->execute();

    if ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $adminName = $row['username'];
    }
}

// Fetch available classes, terms, and years for dropdowns
$classes = sql($DBH, "SELECT DISTINCT class FROM tbl_student_results ORDER BY class", [], "rows");
$terms = sql($DBH, "SELECT DISTINCT term FROM tbl_student_results ORDER BY term", [], "rows");
$years = sql($DBH, "SELECT DISTINCT year FROM tbl_student_results ORDER BY year DESC", [], "rows");

// Initialize variables
$class = $_POST['class'] ?? $_GET['class'] ?? '';
$term = $_POST['term'] ?? $_GET['term'] ?? '';
$year = $_POST['year'] ?? $_GET['year'] ?? '';
$broadsheetData = [];
$subjectList = [];
$studentList = [];
$studentAverages = [];
$subjectAverages = [];
$classAverage = 0;
$studentCount = 0;

// Process form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' || (!empty($class) && !empty($term) && !empty($year))) {
    // Validate inputs
    if (empty($class) || empty($term) || empty($year)) {
        $error = "Please select a class, term, and year.";
    } else {
        try {
            // Get all students in the class (grouped by registration number)
            $studentList = sql($DBH, 
                "SELECT registration_no, MIN(name) as name 
                 FROM tbl_student_results 
                 WHERE class = ? AND term = ? AND year = ? 
                 GROUP BY registration_no 
                 ORDER BY MIN(name)",
                [$class, $term, $year], 
                "rows"
            );
            
            if (empty($studentList)) {
                $error = "No students found for the selected criteria.";
            } else {
                $studentCount = count($studentList);
                
                // Get all subjects for this class
                $subjectList = sql($DBH, 
                    "SELECT DISTINCT subject 
                     FROM tbl_student_results 
                     WHERE class = ? AND term = ? AND year = ? 
                     ORDER BY subject",
                    [$class, $term, $year], 
                    "rows"
                );
                
                // Get all results for this class, term, and year
                $allResults = sql($DBH, 
                    "SELECT * FROM tbl_student_results 
                     WHERE class = ? AND term = ? AND year = ? 
                     ORDER BY registration_no, subject",
                    [$class, $term, $year], 
                    "rows"
                );
                
                // Organize results by student and subject
                foreach ($studentList as $student) {
                    $regNo = $student['registration_no'];
                    $studentAverages[$regNo] = 0;
                    $subjectCount = 0;
                    
                    foreach ($subjectList as $subject) {
                        $subjectName = $subject['subject'];
                        
                        // Find the result for this student and subject
                        $result = array_filter($allResults, function($row) use ($regNo, $subjectName) {
                            return $row['registration_no'] === $regNo && $row['subject'] === $subjectName;
                        });
                        
                        if (!empty($result)) {
                            $result = array_values($result)[0];
                            $total = $result['test1'] + $result['test2'] + $result['exam'];
                            $broadsheetData[$regNo][$subjectName] = [
                                'test1' => $result['test1'],
                                'test2' => $result['test2'],
                                'exam' => $result['exam'],
                                'total' => $total
                            ];
                            
                            $studentAverages[$regNo] += $total;
                            $subjectCount++;
                            
                            // Initialize subject average if not set
                            if (!isset($subjectAverages[$subjectName])) {
                                $subjectAverages[$subjectName] = 0;
                            }
                            $subjectAverages[$subjectName] += $total;
                        } else {
                            $broadsheetData[$regNo][$subjectName] = [
                                'test1' => '-',
                                'test2' => '-',
                                'exam' => '-',
                                'total' => '-'
                            ];
                        }
                    }
                    
                    // Calculate student average
                    if ($subjectCount > 0) {
                        $studentAverages[$regNo] = round($studentAverages[$regNo] / $subjectCount, 2);
                        $classAverage += $studentAverages[$regNo];
                    }
                }
                
                // Calculate subject averages
                foreach ($subjectAverages as $subject => $total) {
                    $subjectAverages[$subject] = round($total / $studentCount, 2);
                }
                
                // Calculate class average
                if ($studentCount > 0) {
                    $classAverage = round($classAverage / $studentCount, 2);
                }
                
                // Calculate positions based on averages
                $studentPositions = calculatePositions($studentAverages);
                
                // Sort student list by position
                usort($studentList, function($a, $b) use ($studentPositions) {
                    $posA = $studentPositions[$a['registration_no']] ?? PHP_INT_MAX;
                    $posB = $studentPositions[$b['registration_no']] ?? PHP_INT_MAX;
                    
                    // If positions are equal, sort by registration number
                    if ($posA == $posB) {
                        return strcmp($a['registration_no'], $b['registration_no']);
                    }
                    
                    return $posA <=> $posB;
                });
            }
        } catch (PDOException $e) {
            $error = "Database error: " . $e->getMessage();
        }
    }
}

// Function to calculate positions based on averages
function calculatePositions($averages) {
    // Filter out non-numeric averages
    $validAverages = array_filter($averages, 'is_numeric');
    
    // Sort averages in descending order
    arsort($validAverages);
    
    $positions = [];
    $position = 1;
    $prevAverage = null;
    $skipCount = 0;
    
    foreach ($validAverages as $regNo => $average) {
        if ($prevAverage !== null && $average < $prevAverage) {
            $position += 1 + $skipCount;
            $skipCount = 0;
        } elseif ($prevAverage !== null && $average == $prevAverage) {
            $skipCount++;
        }
        
        $positions[$regNo] = $position;
        $prevAverage = $average;
    }
    
    return $positions;
}

// Function to determine grade from score
function getGrade($score) {
    if (!is_numeric($score)) return '-';
    
    if ($score >= 90) return 'A+';
    if ($score >= 80) return 'A';
    if ($score >= 75) return 'B+';
    if ($score >= 70) return 'B';
    if ($score >= 65) return 'C+';
    if ($score >= 60) return 'C';
    if ($score >= 50) return 'D';
    return 'F';
}

// Function to format position
function formatPosition($position) {
    if (!is_numeric($position)) return $position;
    
    if ($position == 1) return '1st';
    if ($position == 2) return '2nd';
    if ($position == 3) return '3rd';
    return $position . 'th';
}

// Function to get grade CSS class
function getGradeClass($grade) {
    if ($grade === 'A+') return 'grade-Aplus';
    if ($grade === 'A') return 'grade-A';
    if ($grade === 'B+') return 'grade-Bplus';
    if ($grade === 'B') return 'grade-B';
    if ($grade === 'C+') return 'grade-Cplus';
    if ($grade === 'C') return 'grade-C';
    if ($grade === 'D') return 'grade-D';
    if ($grade === 'F') return 'grade-F';
    return '';
}
?>

<!DOCTYPE html>
<html lang="en">

<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
  <title>VERBUM DEI INT'L COLLEGE - Result Broadsheet</title>
  <link rel="icon" type="image/png" href="https://portal.vda.com.ng/pri_admin/coll_logo.jpg">
  
  <!-- Bootstrap 5 CSS -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
  
  <!-- Font Awesome -->
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
  
  <!-- Google Fonts - Nunito -->
  <link href="https://fonts.googleapis.com/css2?family=Nunito:wght@400;600;700&display=swap" rel="stylesheet">
  
  <!-- Custom CSS -->
  <style>
    :root {
      --primary-color: #1976d2;
      --secondary-color: #64b5f6;
      --accent-color: #bbdefb;
      --dark-color: #0d47a1;
      --light-color: #e3f2fd;
      --text-dark: #212529;
      --text-light: #f8f9fa;
    }
    
    body {
      font-family: 'Nunito', sans-serif;
      background-color: #f5f7fa;
      padding-top: 56px;
      color: var(--text-dark);
      font-size: 0.9rem;
    }
    
    /* Sidebar Styles */
    .sidebar {
      background: linear-gradient(180deg, var(--dark-color) 0%, var(--primary-color) 100%);
      position: fixed;
      top: 56px;
      left: 0;
      width: 250px;
      height: calc(100vh - 56px);
      transition: all 0.3s;
      z-index: 1000;
      overflow-y: auto;
    }
    
    .sidebar .nav-link {
      color: rgba(255, 255, 255, 0.85);
      font-weight: 600;
      padding: 0.75rem 1rem;
      margin-bottom: 0.2rem;
      transition: all 0.2s;
    }
    
    .sidebar .nav-link:hover {
      color: #fff;
      background-color: rgba(255, 255, 255, 0.15);
    }
    
    .sidebar .nav-link.active {
      color: #fff;
      background-color: var(--secondary-color);
      box-shadow: 0 2px 5px rgba(0,0,0,0.1);
    }
    
    .sidebar .nav-link i {
      margin-right: 0.5rem;
      width: 20px;
      text-align: center;
    }
    
    /* Main Content Styles */
    .main-content {
      margin-left: 250px;
      width: calc(100% - 250px);
      transition: all 0.3s;
      min-height: calc(100vh - 56px);
      padding-bottom: 60px;
    }
    
    @media (max-width: 991.98px) {
      .sidebar {
        left: -250px;
      }
      .main-content {
        margin-left: 0;
        width: 100%;
      }
      body {
        padding-top: 56px;
      }
    }
    
    /* Navbar */
    .navbar {
      background-color: var(--primary-color);
      box-shadow: 0 0.15rem 1.75rem 0 rgba(33, 37, 41, 0.1);
      position: fixed;
      top: 0;
      width: 100%;
      z-index: 1030;
      color: white;
    }
    
    /* Welcome banner */
    .welcome-banner {
      background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);
      color: white;
      border-radius: 0.5rem;
      padding: 0.75rem 1rem;
      margin: 1rem;
      box-shadow: 0 4px 6px rgba(0,0,0,0.1);
    }
    
    /* Form Styles */
    .form-container {
      background-color: white;
      border-radius: 0.5rem;
      padding: 1rem;
      margin: 1rem;
      box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.1);
    }
    
    /* Table Styles */
    .table-container {
      background-color: white;
      border-radius: 0.5rem;
      padding: 1rem;
      margin: 1rem;
      box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.1);
      overflow-x: auto;
    }
    
    .broadsheet-table {
      width: 100%;
      border-collapse: collapse;
      font-size: 0.8rem;
    }
    
    .broadsheet-table th,
    .broadsheet-table td {
      padding: 0.3rem;
      border: 1px solid #dee2e6;
      text-align: center;
      vertical-align: middle;
    }
    
    .broadsheet-table th {
      background-color: var(--primary-color);
      color: white;
      font-weight: 600;
      position: sticky;
      top: 0;
      z-index: 10;
      font-size: 0.75rem;
    }
    
    .broadsheet-table .student-name {
      text-align: left;
      font-weight: 600;
      background-color: var(--light-color);
      position: sticky;
      left: 0;
      z-index: 5;
      min-width: 120px;
      max-width: 120px;
      white-space: nowrap;
      overflow: hidden;
      text-overflow: ellipsis;
    }
    
    .broadsheet-table .subject-header {
      background-color: var(--secondary-color);
      writing-mode: vertical-lr;
      transform: rotate(180deg);
      min-width: 25px;
      white-space: nowrap;
      font-size: 0.7rem;
      color: #000000; /* Black text for better visibility */
      font-weight: bold;
    }
    
    /* Alternating background colors for subject headers */
    .broadsheet-table th:nth-child(odd).subject-header {
      background-color: #64b5f6;
    }
    
    .broadsheet-table th:nth-child(even).subject-header {
      background-color: #90caf9;
    }
    
    .broadsheet-table .total-col {
      background-color: #e8f5e9;
      font-weight: 600;
    }
    
    .broadsheet-table .grade-col {
      background-color: #fff3e0;
    }
    
    .broadsheet-table .position-col {
      background-color: #e3f2fd;
      font-weight: 600;
    }
    
    .broadsheet-table .average-row {
      background-color: #f5f5f5;
      font-weight: 600;
    }
    
    .broadsheet-table tr:nth-child(even) {
      background-color: #f9f9f9;
    }
    
    .broadsheet-table tr:hover {
      background-color: #f1f1f1;
    }
    
    /* Footer styles */
    .footer {
      background-color: var(--dark-color);
      color: white;
      padding: 12px 0;
      position: fixed;
      bottom: 0;
      width: 100%;
      z-index: 100;
      font-size: 0.8rem;
    }
    
    .copyright-text {
      font-weight: 600;
    }
    
    .powered-by {
      display: flex;
      align-items: center;
      justify-content: flex-end;
      color: white;
    }
    
    .powered-by img {
      height: 18px;
      margin-left: 6px;
    }
    
    .powered-by-link {
      display: flex;
      align-items: center;
      text-decoration: none;
      color: inherit;
      transition: opacity 0.3s ease;
    }
    
    .powered-by-link:hover {
      opacity: 0.8;
      text-decoration: none;
    }
    
    /* Action buttons */
    .btn-action {
      padding: 0.4rem 0.8rem;
      border-radius: 0.25rem;
      font-weight: 600;
      transition: all 0.3s;
      font-size: 0.85rem;
    }
    
    .btn-primary {
      background-color: var(--primary-color);
      color: white;
      border: none;
    }
    
    .btn-primary:hover {
      background-color: var(--dark-color);
    }
    
    .btn-print {
      background-color: #28a745;
      color: white;
      border: none;
    }
    
    .btn-print:hover {
      background-color: #218838;
    }
    
    /* Grade colors */
    .grade-Aplus { color: #2e7d32; font-weight: bold; }
    .grade-A { color: #388e3c; font-weight: bold; }
    .grade-Bplus { color: #689f38; font-weight: bold; }
    .grade-B { color: #afb42b; font-weight: bold; }
    .grade-Cplus { color: #fbc02d; font-weight: bold; }
    .grade-C { color: #ffa000; font-weight: bold; }
    .grade-D { color: #f57c00; font-weight: bold; }
    .grade-F { color: #d32f2f; font-weight: bold; }
    
    /* Position badges */
    .position-badge {
      display: inline-block;
      padding: 0.2rem 0.5rem;
      border-radius: 0.25rem;
      font-weight: bold;
      font-size: 0.75rem;
    }
    
    .position-1 { background-color: #FFD700; color: #000; }
    .position-2 { background-color: #C0C0C0; color: #000; }
    .position-3 { background-color: #CD7F32; color: #fff; }
    .position-other { background-color: #e9ecef; color: #495057; }
    
    /* Compact form elements */
    .form-select, .form-label {
      font-size: 0.85rem;
    }
    
    /* Print styles */
    @media print {
      body {
        background-color: white;
        font-size: 8pt;
        padding: 0;
        margin: 0;
      }
      
      .no-print {
        display: none !important;
      }
      
      .sidebar {
        display: none !important;
      }
      
      .main-content {
        margin-left: 0;
        width: 100%;
      }
      
      .navbar {
        display: none !important;
      }
      
      .footer {
        display: none !important;
      }
      
      .table-container {
        box-shadow: none;
        padding: 0;
        margin: 0;
      }
      
      .broadsheet-table {
        font-size: 7pt;
      }
      
      .broadsheet-table th {
        -webkit-print-color-adjust: exact;
        print-color-adjust: exact;
      }
      
      .broadsheet-table .student-name {
        min-width: 100px;
        max-width: 100px;
      }
      
      @page {
        size: landscape;
        margin: 0.5cm;
      }
    }
  </style>
</head>

<body>
  <!-- Top Navigation -->
  <nav class="navbar navbar-expand-lg navbar-light">
    <div class="container-fluid">
      <button id="sidebarToggle" class="btn btn-link d-lg-none me-2">
        <i class="fas fa-bars"></i>
      </button>
      <a class="navbar-brand d-none d-lg-block" href="#">
        VERBUM DEI INT'L COLLEGE
      </a>
      <div class="ms-auto d-flex align-items-center">
        <div class="admin-info me-3">
          <span class="admin-name"><?php echo htmlspecialchars($adminName ?? 'Admin'); ?></span>
          <span class="admin-access">(Administrator)</span>
          <i class="fas fa-user-circle admin-icon"></i>
        </div>
        <a href="logout.php" class="btn btn-sm btn-outline-light">
          <i class="fas fa-sign-out-alt"></i> Logout
        </a>
      </div>
    </div>
  </nav>

  <!-- Sidebar -->
  <nav id="sidebar" class="sidebar">
    <div class="position-sticky pt-3">
      <div class="text-center mb-4 d-none d-lg-block">
        <img src="https://portal.vda.com.ng/pri_admin/coll_logo.jpg" alt="School Logo" class="rounded-circle" width="80">
        <h5 class="mt-2 text-white">VERBUM DEI INT'L COLLEGE</h5>
        <p class="text-white-50 small">Admin Dashboard</p>
      </div>
      
      <ul class="nav flex-column">
        <li class="nav-item">
          <a class="nav-link" href="main.php">
            <i class="fas fa-fw fa-tachometer-alt"></i>
            Admin Dashboard
          </a>
        </li>
        <li class="nav-item">
          <a class="nav-link active" href="download_broadsheet.php">
            <i class="fas fa-fw fa-file-pdf"></i>
            Result Broadsheet
          </a>
        </li>
        <li class="nav-item">
          <a class="nav-link" href="uploaded_class_results.php">
            <i class="fas fa-fw fa-list"></i>
            Uploaded Results
          </a>
        </li>
      </ul>
    </div>
  </nav>

  <!-- Main Content -->
  <main class="main-content">
    <!-- Welcome Banner -->
    <div class="welcome-banner">
      <div class="d-flex justify-content-between align-items-center">
        <div>
          <h4 class="mb-1">Result Broadsheet Generator</h4>
          <p class="mb-0 small">VERBUM DEI INT'L COLLEGE Admin Dashboard</p>
        </div>
        <div class="d-none d-sm-inline-block">
          <span class="badge bg-white text-primary p-1 small">
            <i class="fas fa-print me-1"></i> Printable Format
          </span>
        </div>
      </div>
    </div>

    <!-- Search Form -->
    <div class="form-container no-print">
      <form method="POST" action="">
        <div class="row align-items-end">
          <div class="col-md-3">
            <div class="form-group mb-2">
              <label for="class" class="form-label">Class:</label>
              <select id="class" name="class" class="form-select form-select-sm" required>
                <option value="">Select Class</option>
                <?php foreach ($classes as $classOption): ?>
                  <option value="<?php echo htmlspecialchars($classOption['class']); ?>" 
                    <?php echo ($classOption['class'] === $class) ? 'selected' : ''; ?>>
                    <?php echo htmlspecialchars($classOption['class']); ?>
                  </option>
                <?php endforeach; ?>
              </select>
            </div>
          </div>
          
          <div class="col-md-3">
            <div class="form-group mb-2">
              <label for="term" class="form-label">Term:</label>
              <select id="term" name="term" class="form-select form-select-sm" required>
                <option value="">Select Term</option>
                <?php foreach ($terms as $termOption): ?>
                  <option value="<?php echo htmlspecialchars($termOption['term']); ?>" 
                    <?php echo ($termOption['term'] === $term) ? 'selected' : ''; ?>>
                    <?php echo htmlspecialchars($termOption['term']); ?>
                  </option>
                <?php endforeach; ?>
              </select>
            </div>
          </div>
          
          <div class="col-md-3">
            <div class="form-group mb-2">
              <label for="year" class="form-label">Academic Year:</label>
              <select id="year" name="year" class="form-select form-select-sm" required>
                <option value="">Select Year</option>
                <?php foreach ($years as $yearOption): ?>
                  <option value="<?php echo htmlspecialchars($yearOption['year']); ?>" 
                    <?php echo ($yearOption['year'] === $year) ? 'selected' : ''; ?>>
                    <?php echo htmlspecialchars($yearOption['year']); ?>
                  </option>
                <?php endforeach; ?>
              </select>
            </div>
          </div>
          
          <div class="col-md-3">
            <button type="submit" class="btn btn-primary btn-sm w-100">
              <i class="fas fa-search me-1"></i> Generate Broadsheet
            </button>
          </div>
        </div>
      </form>
    </div>

    <?php if (isset($error)): ?>
      <div class="alert alert-danger m-3 py-2">
        <i class="fas fa-exclamation-circle me-2"></i> <?php echo $error; ?>
      </div>
    <?php endif; ?>

    <?php if (!empty($broadsheetData) && !empty($studentList)): ?>
    <div class="table-container">
      <div class="d-flex justify-content-between align-items-center mb-2 no-print">
        <h5 class="mb-0">
          Broadsheet for <?php echo htmlspecialchars($class); ?> - 
          <?php echo htmlspecialchars($term); ?> Term, <?php echo htmlspecialchars($year); ?>
          <small class="text-muted">(<?php echo $studentCount; ?> students)</small>
        </h5>
        <button class="btn btn-print btn-sm" onclick="window.print()">
          <i class="fas fa-print me-1"></i> Print Broadsheet
        </button>
      </div>
      
      <div class="table-responsive">
        <table class="broadsheet-table">
          <thead>
            <tr>
              <th rowspan="2" class="student-name">Student Name</th>
              <th rowspan="2">Reg No</th>
              <?php foreach ($subjectList as $subject): ?>
                <th colspan="3" class="subject-header"><?php echo htmlspecialchars($subject['subject']); ?></th>
              <?php endforeach; ?>
              <th rowspan="2" class="total-col">Avg</th>
              <th rowspan="2" class="grade-col">Grade</th>
              <th rowspan="2" class="position-col">Pos</th>
            </tr>
            <tr>
              <?php foreach ($subjectList as $subject): ?>
                <th>CA1</th>
                <th>CA2</th>
                <th>Total</th>
              <?php endforeach; ?>
            </tr>
          </thead>
          <tbody>
            <?php foreach ($studentList as $student): ?>
              <?php
              $regNo = $student['registration_no'];
              $studentTotal = 0;
              $subjectCount = 0;
              $position = $studentPositions[$regNo] ?? 'N/A';
              $average = $studentAverages[$regNo] ?? '-';
              $grade = getGrade($average);
              $gradeClass = getGradeClass($grade);
              ?>
              <tr>
                <td class="student-name" title="<?php echo htmlspecialchars($student['name']); ?>">
                  <?php echo htmlspecialchars($student['name']); ?>
                </td>
                <td><?php echo htmlspecialchars($regNo); ?></td>
                
                <?php foreach ($subjectList as $subject): ?>
                  <?php
                  $subjectName = $subject['subject'];
                  $result = $broadsheetData[$regNo][$subjectName] ?? null;
                  
                  if ($result && is_numeric($result['total'])) {
                    $studentTotal += $result['total'];
                    $subjectCount++;
                  }
                  ?>
                  
                  <td><?php echo $result ? htmlspecialchars($result['test1']) : '-'; ?></td>
                  <td><?php echo $result ? htmlspecialchars($result['test2']) : '-'; ?></td>
                  <td class="total-col">
                    <?php echo $result ? htmlspecialchars($result['total']) : '-'; ?>
                  </td>
                <?php endforeach; ?>
                
                <td class="total-col">
                  <?php echo $average; ?>
                </td>
                
                <td class="grade-col <?php echo $gradeClass; ?>">
                  <?php echo $grade; ?>
                </td>
                
                <td class="position-col">
                  <?php if (is_numeric($position)): ?>
                    <?php
                    $positionClass = 'position-other';
                    if ($position == 1) $positionClass = 'position-1';
                    elseif ($position == 2) $positionClass = 'position-2';
                    elseif ($position == 3) $positionClass = 'position-3';
                    ?>
                    <span class="position-badge <?php echo $positionClass; ?>">
                      <?php echo formatPosition($position); ?>
                    </span>
                  <?php else: ?>
                    <?php echo $position; ?>
                  <?php endif; ?>
                </td>
              </tr>
            <?php endforeach; ?>
            
            <!-- Subject Averages Row -->
            <tr class="average-row">
              <td colspan="2" class="student-name">Subject Averages</td>
              
              <?php foreach ($subjectList as $subject): ?>
                <?php
                $subjectName = $subject['subject'];
                $avg = $subjectAverages[$subjectName] ?? '-';
                ?>
                <td colspan="2"></td>
                <td class="total-col"><?php echo $avg; ?></td>
              <?php endforeach; ?>
              
              <td class="total-col"><?php echo $classAverage; ?></td>
              <td colspan="2"></td>
            </tr>
          </tbody>
        </table>
      </div>
      
      <div class="mt-2 no-print">
        <div class="alert alert-info py-2">
          <h6 class="mb-1"><i class="fas fa-info-circle me-1"></i> Grading System</h6>
          <div class="row">
            <div class="col-md-2"><span class="grade-Aplus">A+</span>: 90-100%</div>
            <div class="col-md-2"><span class="grade-A">A</span>: 80-89%</div>
            <div class="col-md-2"><span class="grade-Bplus">B+</span>: 75-79%</div>
            <div class="col-md-2"><span class="grade-B">B</span>: 70-74%</div>
            <div class="col-md-2"><span class="grade-Cplus">C+</span>: 65-69%</div>
            <div class="col-md-2"><span class="grade-C">C</span>: 60-64%</div>
            <div class="col-md-2"><span class="grade-D">D</span>: 50-59%</div>
            <div class="col-md-2"><span class="grade-F">F</span>: 0-49%</div>
          </div>
        </div>
        
        <button class="btn btn-print btn-sm" onclick="window.print()">
          <i class="fas fa-print me-1"></i> Print Broadsheet
        </button>
      </div>
    </div>
    <?php elseif ($_SERVER['REQUEST_METHOD'] === 'POST'): ?>
      <div class="form-container text-center py-3">
        <i class="fas fa-info-circle fa-2x mb-2 text-muted"></i>
        <h5 class="text-muted">No Results Found</h5>
        <p class="mb-0">There are no results for the selected class, term, and year combination.</p>
      </div>
    <?php else: ?>
      <div class="form-container text-center py-4">
        <i class="fas fa-file-pdf fa-3x mb-2 text-muted"></i>
        <h5 class="text-muted">Result Broadsheet Generator</h5>
        <p class="mb-0">Select a class, term, and academic year to generate a broadsheet.</p>
      </div>
    <?php endif; ?>
  </main>

  <!-- Footer -->
  <footer class="footer no-print">
    <div class="container-fluid">
      <div class="row align-items-center">
        <div class="col-md-6">
          <span class="copyright-text">Copyright &copy; VERBUM DEI INT'L COLLEGE <?php echo date("Y"); ?></span>
        </div>
        <div class="col-md-6">
          <div class="powered-by">
            <a href="https://dataswift.com.ng" target="_blank" rel="noopener noreferrer" class="powered-by-link">
              <span>Powered by DataSwift</span>
              <img src="datalogo.png" alt="DataSwift Logo">
            </a>
          </div>
        </div>
      </div>
    </div>
  </footer>

  <!-- Bootstrap 5 JS Bundle with Popper --> 
  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
  
  <!-- Custom Scripts -->
  <script>
    // Toggle sidebar on mobile
    document.getElementById('sidebarToggle').addEventListener('click', function() {
      document.getElementById('sidebar').classList.toggle('active');
      document.querySelector('.main-content').classList.toggle('active');
    });
    
    // Close sidebar when clicking outside on mobile
    document.addEventListener('click', function(event) {
      const sidebar = document.getElementById('sidebar');
      const sidebarToggle = document.getElementById('sidebarToggle');
      
      if (window.innerWidth < 992 && 
          !sidebar.contains(event.target) && 
          !sidebarToggle.contains(event.target) && 
          sidebar.classList.contains('active')) {
        sidebar.classList.remove('active');
        document.querySelector('.main-content').classList.remove('active');
      }
    });
  </script>
</body>
</html>