<?php
require_once('source/session.php');
require_once('source/error.php');
require_once('source/dbconfig.php');
require_once('source/function.php');
session_start();

// Check if the user is logged in
if (isset($_SESSION['SESS_ID_ADMIN'])) {
    $userId = $_SESSION['SESS_ID_ADMIN'];

    // Fetch user details based on the ID
    $query = "SELECT * FROM tbl_admin WHERE id = :id";
    $stmt = $DBH->prepare($query);
    $stmt->bindParam(':id', $userId, PDO::PARAM_INT);
    $stmt->execute();

    if ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $adminName = $row['username'];
    }
}

// Fetch dynamic options for class, term, year, and subject from the database
$classes = sql($DBH, "SELECT DISTINCT class FROM tbl_student_results ORDER BY class", [], "rows");
$terms = sql($DBH, "SELECT DISTINCT term FROM tbl_student_results ORDER BY term", [], "rows");
$years = sql($DBH, "SELECT DISTINCT year FROM tbl_student_results ORDER BY year DESC", [], "rows");
$subjects = sql($DBH, "SELECT DISTINCT subject FROM tbl_student_results ORDER BY subject", [], "rows");

// Initialize variables to store class, term, and year for display
$page_title = '';
$subject_name = '';
$total_results_count = 0;
$class = $_POST['class'] ?? '';
$term = $_POST['term'] ?? '';
$year = $_POST['year'] ?? '';
$subject = $_POST['subject'] ?? '';
$score_records = [];

if ($_SERVER["REQUEST_METHOD"] == "POST" && !isset($_POST['update_results'])) {
    // Extract search data from the form
    $class = strip_tags($_POST['class']);
    $term = strip_tags($_POST['term']);
    $year = strip_tags($_POST['year']);
    $subject = strip_tags($_POST['subject']);

    // Retrieve score records from the database based on the search criteria
    $score_records = sql($DBH, 
        "SELECT * FROM tbl_student_results 
         WHERE class = ? AND term = ? AND year = ? AND subject = ? 
         ORDER BY name",
        [$class, $term, $year, $subject], 
        "rows"
    );

    // Set page title
    $page_title = "Class: $class, Term: $term, Year: $year";

    // Set subject name
    $subject_name = $subject;

    // Count of total student results
    $total_results_count = count($score_records);
}

// Handle form submission for updating scores
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['update_results'])) {
    // Extract data from the form
    $class = strip_tags($_POST['class']);
    $term = strip_tags($_POST['term']);
    $year = strip_tags($_POST['year']);
    $subject = strip_tags($_POST['subject']);
    $record_ids = $_POST['record_id'];
    $test1_scores = $_POST['test1'];
    $test2_scores = $_POST['test2'];
    $exam_scores = $_POST['exam'];

    // Update the records in the database
    $update_query = "UPDATE tbl_student_results SET test1 = ?, test2 = ?, exam = ?, total = (test1 + test2 + exam) WHERE id = ?";
    
    $success_count = 0;
    $error_messages = [];
    
    foreach ($record_ids as $index => $record_id) {
        $test1 = floatval($test1_scores[$index]);
        $test2 = floatval($test2_scores[$index]);
        $exam = floatval($exam_scores[$index]);
        
        // Validate scores
        if ($test1 >= 0 && $test2 >= 0 && $exam >= 0) {
            $stmt = $DBH->prepare($update_query);
            if ($stmt->execute([$test1, $test2, $exam, $record_id])) {
                $success_count++;
            } else {
                $error_messages[] = "Failed to update record ID: $record_id";
            }
        } else {
            $error_messages[] = "Invalid score values for record ID: $record_id";
        }
    }

    // Refresh the records after update
    $score_records = sql($DBH, 
        "SELECT * FROM tbl_student_results 
         WHERE class = ? AND term = ? AND year = ? AND subject = ? 
         ORDER BY name",
        [$class, $term, $year, $subject], 
        "rows"
    );
    
    $total_results_count = count($score_records);
    
    if ($success_count > 0 && empty($error_messages)) {
        $success_message = "Successfully updated $success_count scores for $subject - $class, $term, $year";
    } elseif (!empty($error_messages)) {
        $error_message = implode("<br>", $error_messages);
    } else {
        $error_message = "No scores were updated.";
    }
}
?>

<!DOCTYPE html>
<html lang="en">

<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
  <title>VERBUM DEI ACADEMY - Edit Subject Scores</title>
  <link rel="icon" type="image/png" href="https://portal.vda.com.ng/pri_admin/pri_logo.jpg">
  
  <!-- Bootstrap 5 CSS -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
  
  <!-- Font Awesome -->
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
  
  <!-- Google Fonts - Nunito -->
  <link href="https://fonts.googleapis.com/css2?family=Nunito:wght@400;600;700&display=swap" rel="stylesheet">
  
  <!-- Custom CSS -->
  <style>
    :root {
      --primary-color: #1976d2;
      --secondary-color: #64b5f6;
      --accent-color: #bbdefb;
      --dark-color: #0d47a1;
      --light-color: #e3f2fd;
      --text-dark: #212529;
      --text-light: #f8f9fa;
    }
    
    body {
      font-family: 'Nunito', sans-serif;
      background-color: #f5f7fa;
      padding-top: 56px;
      color: var(--text-dark);
      font-size: 0.9rem;
    }
    
    /* Sidebar Styles */
    .sidebar {
      background: linear-gradient(180deg, var(--dark-color) 0%, var(--primary-color) 100%);
      position: fixed;
      top: 56px;
      left: 0;
      width: 250px;
      height: calc(100vh - 56px);
      transition: all 0.3s;
      z-index: 1000;
      overflow-y: auto;
    }
    
    .sidebar .nav-link {
      color: rgba(255, 255, 255, 0.85);
      font-weight: 600;
      padding: 0.75rem 1rem;
      margin-bottom: 0.2rem;
      transition: all 0.2s;
    }
    
    .sidebar .nav-link:hover {
      color: #fff;
      background-color: rgba(255, 255, 255, 0.15);
    }
    
    .sidebar .nav-link.active {
      color: #fff;
      background-color: var(--secondary-color);
      box-shadow: 0 2px 5px rgba(0,0,0,0.1);
    }
    
    .sidebar .nav-link i {
      margin-right: 0.5rem;
      width: 20px;
      text-align: center;
    }
    
    /* Main Content Styles */
    .main-content {
      margin-left: 250px;
      width: calc(100% - 250px);
      transition: all 0.3s;
      min-height: calc(100vh - 56px);
      padding-bottom: 60px;
    }
    
    @media (max-width: 991.98px) {
      .sidebar {
        left: -250px;
      }
      .main-content {
        margin-left: 0;
        width: 100%;
      }
      body {
        padding-top: 56px;
      }
    }
    
    /* Navbar */
    .navbar {
      background-color: var(--primary-color);
      box-shadow: 0 0.15rem 1.75rem 0 rgba(33, 37, 41, 0.1);
      position: fixed;
      top: 0;
      width: 100%;
      z-index: 1030;
      color: white;
    }
    
    /* Welcome banner */
    .welcome-banner {
      background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);
      color: white;
      border-radius: 0.5rem;
      padding: 0.75rem 1rem;
      margin: 1rem;
      box-shadow: 0 4px 6px rgba(0,0,0,0.1);
    }
    
    /* Form Styles */
    .form-container {
      background-color: white;
      border-radius: 0.5rem;
      padding: 1rem;
      margin: 1rem;
      box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.1);
    }
    
    /* Table Styles */
    .table-container {
      background-color: white;
      border-radius: 0.5rem;
      padding: 1rem;
      margin: 1rem;
      box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.1);
      overflow-x: auto;
    }
    
    .scores-table {
      width: 100%;
      border-collapse: collapse;
      font-size: 0.85rem;
    }
    
    .scores-table th,
    .scores-table td {
      padding: 0.5rem;
      border: 1px solid #dee2e6;
      text-align: center;
      vertical-align: middle;
    }
    
    .scores-table th {
      background-color: var(--primary-color);
      color: white;
      font-weight: 600;
      position: sticky;
      top: 0;
      z-index: 10;
    }
    
    .scores-table input {
      width: 70px;
      padding: 0.25rem 0.5rem;
      border: 1px solid #ced4da;
      border-radius: 0.25rem;
      text-align: center;
      font-size: 0.85rem;
    }
    
    .scores-table input:focus {
      border-color: var(--secondary-color);
      box-shadow: 0 0 0 0.2rem rgba(100, 181, 246, 0.25);
      outline: none;
    }
    
    .scores-table .student-name {
      text-align: left;
      font-weight: 600;
      background-color: var(--light-color);
    }
    
    .scores-table .total-col {
      background-color: #e8f5e9;
      font-weight: 600;
    }
    
    .scores-table tr:nth-child(even) {
      background-color: #f9f9f9;
    }
    
    .scores-table tr:hover {
      background-color: #f1f1f1;
    }
    
    /* Footer styles */
    .footer {
      background-color: var(--dark-color);
      color: white;
      padding: 12px 0;
      position: fixed;
      bottom: 0;
      width: 100%;
      z-index: 100;
      font-size: 0.8rem;
    }
    
    .copyright-text {
      font-weight: 600;
    }
    
    .powered-by {
      display: flex;
      align-items: center;
      justify-content: flex-end;
      color: white;
    }
    
    .powered-by img {
      height: 18px;
      margin-left: 6px;
    }
    
    .powered-by-link {
      display: flex;
      align-items: center;
      text-decoration: none;
      color: inherit;
      transition: opacity 0.3s ease;
    }
    
    .powered-by-link:hover {
      opacity: 0.8;
      text-decoration: none;
    }
    
    /* Action buttons */
    .btn-action {
      padding: 0.5rem 1rem;
      border-radius: 0.25rem;
      font-weight: 600;
      transition: all 0.3s;
      font-size: 0.85rem;
    }
    
    .btn-primary {
      background-color: var(--primary-color);
      color: white;
      border: none;
    }
    
    .btn-primary:hover {
      background-color: var(--dark-color);
    }
    
    .btn-success {
      background-color: #28a745;
      color: white;
      border: none;
    }
    
    .btn-success:hover {
      background-color: #218838;
    }
    
    .btn-danger {
      background-color: #dc3545;
      color: white;
      border: none;
    }
    
    .btn-danger:hover {
      background-color: #c82333;
    }
    
    /* Alert styles */
    .alert {
      padding: 0.75rem 1rem;
      border-radius: 0.25rem;
      margin-bottom: 1rem;
    }
    
    .alert-success {
      background-color: #d4edda;
      border-color: #c3e6cb;
      color: #155724;
    }
    
    .alert-danger {
      background-color: #f8d7da;
      border-color: #f5c6cb;
      color: #721c24;
    }
    
    /* Info cards */
    .info-card {
      background: linear-gradient(135deg, var(--light-color) 0%, #ffffff 100%);
      border-left: 4px solid var(--primary-color);
      padding: 0.75rem;
      margin-bottom: 0.5rem;
      border-radius: 0 0.25rem 0.25rem 0;
    }
    
    /* Compact form elements */
    .form-select, .form-label {
      font-size: 0.85rem;
    }
  </style>
</head>

<body>
  <!-- Top Navigation -->
  <nav class="navbar navbar-expand-lg navbar-light">
    <div class="container-fluid">
      <button id="sidebarToggle" class="btn btn-link d-lg-none me-2">
        <i class="fas fa-bars"></i>
      </button>
      <a class="navbar-brand d-none d-lg-block" href="#">
        VERBUM DEI ACADEMY
      </a>
      <div class="ms-auto d-flex align-items-center">
        <div class="admin-info me-3">
          <span class="admin-name"><?php echo htmlspecialchars($adminName ?? 'Admin'); ?></span>
          <span class="admin-access">(Administrator)</span>
          <i class="fas fa-user-circle admin-icon"></i>
        </div>
        <a href="logout.php" class="btn btn-sm btn-outline-light">
          <i class="fas fa-sign-out-alt"></i> Logout
        </a>
      </div>
    </div>
  </nav>

  <!-- Sidebar -->
  <nav id="sidebar" class="sidebar">
    <div class="position-sticky pt-3">
      <div class="text-center mb-4 d-none d-lg-block">
        <img src="https://portal.vda.com.ng/pri_admin/pri_logo.jpg" alt="School Logo" class="rounded-circle" width="80">
        <h5 class="mt-2 text-white">VERBUM DEI ACADEMY</h5>
        <p class="text-white-50 small">Admin Dashboard</p>
      </div>
      
      <ul class="nav flex-column">
        <li class="nav-item">
          <a class="nav-link" href="main.php">
            <i class="fas fa-fw fa-tachometer-alt"></i>
            Admin Dashboard
          </a>
        </li>
        <li class="nav-item">
          <a class="nav-link active" href="edit_scores_subject.php">
            <i class="fas fa-fw fa-edit"></i>
            Edit Subject Scores
          </a>
        </li>
        <li class="nav-item">
          <a class="nav-link" href="download_broadsheet.php">
            <i class="fas fa-fw fa-file-pdf"></i>
            Result Broadsheet
          </a>
        </li>
        <li class="nav-item">
          <a class="nav-link" href="uploaded_class_results.php">
            <i class="fas fa-fw fa-list"></i>
            Uploaded Results
          </a>
        </li>
      </ul>
    </div>
  </nav>

  <!-- Main Content -->
  <main class="main-content">
    <!-- Welcome Banner -->
    <div class="welcome-banner">
      <div class="d-flex justify-content-between align-items-center">
        <div>
          <h4 class="mb-1">Edit Subject Scores</h4>
          <p class="mb-0 small">VERBUM DEI ACADEMY Admin Dashboard</p>
        </div>
        <div class="d-none d-sm-inline-block">
          <span class="badge bg-white text-primary p-1 small">
            <i class="fas fa-edit me-1"></i> Edit Scores
          </span>
        </div>
      </div>
    </div>

    <!-- Search Form -->
    <div class="form-container">
      <form method="POST" action="">
        <div class="row align-items-end">
          <div class="col-md-3">
            <div class="form-group mb-2">
              <label for="class" class="form-label">Class:</label>
              <select id="class" name="class" class="form-select form-select-sm" required>
                <option value="">Select Class</option>
                <?php foreach ($classes as $classOption): ?>
                  <option value="<?php echo htmlspecialchars($classOption['class']); ?>" 
                    <?php echo ($classOption['class'] === $class) ? 'selected' : ''; ?>>
                    <?php echo htmlspecialchars($classOption['class']); ?>
                  </option>
                <?php endforeach; ?>
              </select>
            </div>
          </div>
          
          <div class="col-md-2">
            <div class="form-group mb-2">
              <label for="term" class="form-label">Term:</label>
              <select id="term" name="term" class="form-select form-select-sm" required>
                <option value="">Select Term</option>
                <?php foreach ($terms as $termOption): ?>
                  <option value="<?php echo htmlspecialchars($termOption['term']); ?>" 
                    <?php echo ($termOption['term'] === $term) ? 'selected' : ''; ?>>
                    <?php echo htmlspecialchars($termOption['term']); ?>
                  </option>
                <?php endforeach; ?>
              </select>
            </div>
          </div>
          
          <div class="col-md-2">
            <div class="form-group mb-2">
              <label for="year" class="form-label">Year:</label>
              <select id="year" name="year" class="form-select form-select-sm" required>
                <option value="">Select Year</option>
                <?php foreach ($years as $yearOption): ?>
                  <option value="<?php echo htmlspecialchars($yearOption['year']); ?>" 
                    <?php echo ($yearOption['year'] === $year) ? 'selected' : ''; ?>>
                    <?php echo htmlspecialchars($yearOption['year']); ?>
                  </option>
                <?php endforeach; ?>
              </select>
            </div>
          </div>
          
          <div class="col-md-3">
            <div class="form-group mb-2">
              <label for="subject" class="form-label">Subject:</label>
              <select id="subject" name="subject" class="form-select form-select-sm" required>
                <option value="">Select Subject</option>
                <?php foreach ($subjects as $subjectOption): ?>
                  <option value="<?php echo htmlspecialchars($subjectOption['subject']); ?>" 
                    <?php echo ($subjectOption['subject'] === $subject) ? 'selected' : ''; ?>>
                    <?php echo htmlspecialchars($subjectOption['subject']); ?>
                  </option>
                <?php endforeach; ?>
              </select>
            </div>
          </div>
          
          <div class="col-md-2">
            <button type="submit" class="btn btn-primary btn-sm w-100">
              <i class="fas fa-search me-1"></i> Search
            </button>
          </div>
        </div>
      </form>
    </div>

    <?php if (isset($error_message)): ?>
      <div class="alert alert-danger m-3 py-2">
        <i class="fas fa-exclamation-circle me-2"></i> <?php echo $error_message; ?>
      </div>
    <?php endif; ?>

    <?php if (isset($success_message)): ?>
      <div class="alert alert-success m-3 py-2">
        <i class="fas fa-check-circle me-2"></i> <?php echo $success_message; ?>
      </div>
    <?php endif; ?>

    <?php if (!empty($subject_name) && !empty($page_title)): ?>
      <div class="form-container">
        <div class="row">
          <div class="col-md-6">
            <div class="info-card">
              <h6 class="mb-1"><i class="fas fa-book me-2"></i>Subject: <?php echo htmlspecialchars($subject_name); ?></h6>
              <p class="mb-0 small"><?php echo htmlspecialchars($page_title); ?></p>
            </div>
          </div>
          <div class="col-md-6">
            <div class="info-card">
              <h6 class="mb-1"><i class="fas fa-users me-2"></i>Student Count: <?php echo $total_results_count; ?></h6>
              <p class="mb-0 small">Total records found for this selection</p>
            </div>
          </div>
        </div>
      </div>
    <?php endif; ?>

    <?php if (!empty($score_records)): ?>
      <div class="table-container">
        <form method="POST">
          <input type="hidden" name="class" value="<?php echo htmlspecialchars($class); ?>">
          <input type="hidden" name="term" value="<?php echo htmlspecialchars($term); ?>">
          <input type="hidden" name="year" value="<?php echo htmlspecialchars($year); ?>">
          <input type="hidden" name="subject" value="<?php echo htmlspecialchars($subject); ?>">
          
          <div class="d-flex justify-content-between align-items-center mb-3">
            <h5 class="mb-0">
              Edit Scores for <?php echo htmlspecialchars($subject); ?> 
              <small class="text-muted">(<?php echo $total_results_count; ?> students)</small>
            </h5>
            <button type="submit" name="update_results" class="btn btn-success btn-sm">
              <i class="fas fa-save me-1"></i> Save All Changes
            </button>
          </div>
          
          <div class="table-responsive">
            <table class="scores-table">
              <thead>
                <tr>
                  <th width="5%">S/N</th>
                  <th>Student Name</th>
                  <th>Admission No.</th>
                  <th>CA1</th>
                  <th>CA2</th>
                  <th>Exam</th>
                  <th class="total-col">Total</th>
                  <th width="10%">Action</th>
                </tr>
              </thead>
              <tbody>
                <?php foreach ($score_records as $index => $record): ?>
                  <tr>
                    <td><?php echo $index + 1; ?></td>
                    <td class="student-name"><?php echo htmlspecialchars($record['name']); ?></td>
                    <td><?php echo htmlspecialchars($record['registration_no']); ?></td>
                    <td>
                      <input type="number" name="test1[]" value="<?php echo $record['test1']; ?>" 
                             min="0" max="100" step="0.5" required>
                    </td>
                    <td>
                      <input type="number" name="test2[]" value="<?php echo $record['test2']; ?>" 
                             min="0" max="100" step="0.5" required>
                    </td>
                    <td>
                      <input type="number" name="exam[]" value="<?php echo $record['exam']; ?>" 
                             min="0" max="100" step="0.5" required>
                    </td>
                    <td class="total-col"><?php echo $record['total']; ?></td>
                    <td>
                      <input type="hidden" name="record_id[]" value="<?php echo $record['id']; ?>">
                      <a href="delete_result_subject.php?record_id=<?php echo $record['id']; ?>&class=<?php echo urlencode($class); ?>&term=<?php echo urlencode($term); ?>&year=<?php echo urlencode($year); ?>&subject=<?php echo urlencode($subject); ?>" 
                         class="btn btn-danger btn-sm py-0 px-2"
                         onclick="return confirm('Are you sure you want to delete this score record?')">
                        <i class="fas fa-trash"></i>
                      </a>
                    </td>
                  </tr>
                <?php endforeach; ?>
              </tbody>
            </table>
          </div>
          
          <div class="mt-3 text-center">
            <button type="submit" name="update_results" class="btn btn-success btn-action">
              <i class="fas fa-save me-1"></i> Save All Updated Scores
            </button>
            <button type="button" class="btn btn-secondary btn-action ms-2" onclick="resetForm()">
              <i class="fas fa-undo me-1"></i> Reset Changes
            </button>
          </div>
        </form>
        
        <div class="mt-3">
          <div class="alert alert-info py-2">
            <h6 class="mb-1"><i class="fas fa-info-circle me-1"></i> Instructions</h6>
            <p class="mb-0 small">
              1. Edit scores in the CA1, CA2, and Exam columns.<br>
              2. Total scores are automatically calculated.<br>
              3. Click "Save All Updated Scores" to apply changes.<br>
              4. Use the delete button to remove individual records.
            </p>
          </div>
        </div>
      </div>
    <?php elseif ($_SERVER["REQUEST_METHOD"] == "POST" && empty($score_records)): ?>
      <div class="form-container text-center py-4">
        <i class="fas fa-search fa-3x mb-2 text-muted"></i>
        <h5 class="text-muted">No Records Found</h5>
        <p class="mb-0">There are no score records for the selected criteria.</p>
      </div>
    <?php else: ?>
      <div class="form-container text-center py-4">
        <i class="fas fa-edit fa-3x mb-2 text-muted"></i>
        <h5 class="text-muted">Edit Subject Scores</h5>
        <p class="mb-0">Select a class, term, year, and subject to edit scores.</p>
      </div>
    <?php endif; ?>
  </main>

  <!-- Footer -->
  <footer class="footer">
    <div class="container-fluid">
      <div class="row align-items-center">
        <div class="col-md-6">
          <span class="copyright-text">Copyright &copy; VERBUM DEI ACADEMY <?php echo date("Y"); ?></span>
        </div>
        <div class="col-md-6">
          <div class="powered-by">
            <a href="https://dataswift.com.ng" target="_blank" rel="noopener noreferrer" class="powered-by-link">
              <span>Powered by DataSwift</span>
              <img src="datalogo.png" alt="DataSwift Logo">
            </a>
          </div>
        </div>
      </div>
    </div>
  </footer>

  <!-- Bootstrap 5 JS Bundle with Popper -->
  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
  
  <!-- Custom Scripts -->
  <script>
    // Toggle sidebar on mobile
    document.getElementById('sidebarToggle').addEventListener('click', function() {
      document.getElementById('sidebar').classList.toggle('active');
      document.querySelector('.main-content').classList.toggle('active');
    });
    
    // Close sidebar when clicking outside on mobile
    document.addEventListener('click', function(event) {
      const sidebar = document.getElementById('sidebar');
      const sidebarToggle = document.getElementById('sidebarToggle');
      
      if (window.innerWidth < 992 && 
          !sidebar.contains(event.target) && 
          !sidebarToggle.contains(event.target) && 
          sidebar.classList.contains('active')) {
        sidebar.classList.remove('active');
        document.querySelector('.main-content').classList.remove('active');
      }
    });
    
    // Auto-calculate total on input change
    document.querySelectorAll('input[type="number"]').forEach(input => {
      input.addEventListener('input', function() {
        const row = this.closest('tr');
        const test1 = parseFloat(row.querySelector('input[name="test1[]"]').value) || 0;
        const test2 = parseFloat(row.querySelector('input[name="test2[]"]').value) || 0;
        const exam = parseFloat(row.querySelector('input[name="exam[]"]').value) || 0;
        const totalCell = row.querySelector('.total-col');
        
        const total = test1 + test2 + exam;
        totalCell.textContent = total.toFixed(1);
      });
    });
    
    // Reset form function
    function resetForm() {
      if (confirm('Are you sure you want to reset all changes?')) {
        location.reload();
      }
    }
    
    // Focus first input field if records exist
    <?php if (!empty($score_records)): ?>
      document.addEventListener('DOMContentLoaded', function() {
        const firstInput = document.querySelector('input[type="number"]');
        if (firstInput) {
          firstInput.focus();
        }
      });
    <?php endif; ?>
  </script>
</body>
</html>