<?php
try {
    require_once('source/session.php');
    require_once('source/error.php');
    require_once('source/dbconfig.php');
    require_once('source/function.php');
    session_start();

    // Check if the user is logged in
    if (isset($_SESSION['SESS_ID_ADMIN'])) {
        $userId = $_SESSION['SESS_ID_ADMIN'];

        // Fetch user details based on the ID
        $query = "SELECT * FROM tbl_admin WHERE id = :id";
        $stmt = $DBH->prepare($query);
        $stmt->bindParam(':id', $userId, PDO::PARAM_INT);
        $stmt->execute();

        if ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $adminName = $row['username'];
        }
    }

    // Fetch dynamic options for class, term, and year from the database
    $classes = sql($DBH, "SELECT DISTINCT class FROM skills ORDER BY class", [], "rows");
    $terms = sql($DBH, "SELECT DISTINCT term FROM skills ORDER BY term", [], "rows");
    $years = sql($DBH, "SELECT DISTINCT year FROM skills ORDER BY year DESC", [], "rows");
    
    // Fetch teachers for filtering
    $teachers = sql($DBH, "SELECT DISTINCT teacher_no FROM skills ORDER BY teacher_no", [], "rows");

    // Initialize variables
    $class = $_POST['class'] ?? '';
    $term = $_POST['term'] ?? '';
    $year = $_POST['year'] ?? '';
    $teacher_no = $_POST['teacher_no'] ?? '';
    $page_title = '';
    $total_results_count = 0;
    $skills_results = [];

    if ($_SERVER["REQUEST_METHOD"] == "POST") {
        // Extract search data from the form
        $class = strip_tags($_POST['class'] ?? '');
        $term = strip_tags($_POST['term'] ?? '');
        $year = strip_tags($_POST['year'] ?? '');
        $teacher_no = strip_tags($_POST['teacher_no'] ?? '');

        // Build WHERE conditions dynamically
        $whereConditions = [];
        $params = [];
        
        if (!empty($class)) {
            $whereConditions[] = "class = ?";
            $params[] = $class;
        }
        
        if (!empty($term)) {
            $whereConditions[] = "term = ?";
            $params[] = $term;
        }
        
        if (!empty($year)) {
            $whereConditions[] = "year = ?";
            $params[] = $year;
        }
        
        if (!empty($teacher_no)) {
            $whereConditions[] = "teacher_no = ?";
            $params[] = $teacher_no;
        }
        
        // Only execute query if we have at least one filter
        if (!empty($whereConditions)) {
            $whereClause = "WHERE " . implode(" AND ", $whereConditions);
            
            // Retrieve skills records from the database based on the search criteria
            $skills_results = sql($DBH, 
                "SELECT id, name, registration_no, class, subclass, year, term, teacher_no,
                        Creativity, Fluency, Honesty, Leadership, 
                        Neatness, Politeness, Punctuality, SelfControl, 
                        SenseOfResponsibility, Sociability, 
                        created_at, updated_at
                 FROM skills 
                 $whereClause 
                 ORDER BY name, registration_no",
                $params, 
                "rows"
            );

            // Set page title
            $page_title = "Psychomotor Skills Modification";

            // Count of total records
            $total_results_count = count($skills_results);

            // Handle form submissions for saving all fields
            if (isset($_POST['save_all'])) {
                $ids = $_POST['id'];
                $creativity_values = $_POST['creativity'];
                $fluency_values = $_POST['fluency'];
                $honesty_values = $_POST['honesty'];
                $leadership_values = $_POST['leadership'];
                $neatness_values = $_POST['neatness'];
                $politeness_values = $_POST['politeness'];
                $punctuality_values = $_POST['punctuality'];
                $selfcontrol_values = $_POST['selfcontrol'];
                $senseofresponsibility_values = $_POST['senseofresponsibility'];
                $sociability_values = $_POST['sociability'];

                $success_count = 0;
                $error_messages = [];
                
                foreach ($ids as $index => $id) {
                    $creativity = intval($creativity_values[$index]);
                    $fluency = intval($fluency_values[$index]);
                    $honesty = intval($honesty_values[$index]);
                    $leadership = intval($leadership_values[$index]);
                    $neatness = intval($neatness_values[$index]);
                    $politeness = intval($politeness_values[$index]);
                    $punctuality = intval($punctuality_values[$index]);
                    $selfcontrol = intval($selfcontrol_values[$index]);
                    $senseofresponsibility = intval($senseofresponsibility_values[$index]);
                    $sociability = intval($sociability_values[$index]);
                    
                    // Validate ratings (1-5)
                    $ratings = [
                        $creativity, $fluency, $honesty, $leadership,
                        $neatness, $politeness, $punctuality, $selfcontrol,
                        $senseofresponsibility, $sociability
                    ];
                    
                    $valid_ratings = true;
                    foreach ($ratings as $rating) {
                        if ($rating < 1 || $rating > 5) {
                            $valid_ratings = false;
                            break;
                        }
                    }
                    
                    if ($valid_ratings) {
                        $update_query = "UPDATE skills SET 
                                        Creativity = ?, 
                                        Fluency = ?, 
                                        Honesty = ?, 
                                        Leadership = ?, 
                                        Neatness = ?, 
                                        Politeness = ?, 
                                        Punctuality = ?, 
                                        SelfControl = ?, 
                                        SenseOfResponsibility = ?, 
                                        Sociability = ?,
                                        updated_at = NOW()
                                        WHERE id = ?";
                        
                        $stmt = $DBH->prepare($update_query);
                        if ($stmt->execute([
                            $creativity, $fluency, $honesty, $leadership,
                            $neatness, $politeness, $punctuality, $selfcontrol,
                            $senseofresponsibility, $sociability, $id
                        ])) {
                            $success_count++;
                        } else {
                            $error_messages[] = "Failed to update record ID: $id";
                        }
                    } else {
                        $error_messages[] = "Invalid rating values for record ID: $id (must be 1-5)";
                    }
                }

                if ($success_count > 0 && empty($error_messages)) {
                    $success_message = "Successfully updated $success_count skills records";
                    
                    // Refresh the results
                    $skills_results = sql($DBH, 
                        "SELECT id, name, registration_no, class, subclass, year, term, teacher_no,
                                Creativity, Fluency, Honesty, Leadership, 
                                Neatness, Politeness, Punctuality, SelfControl, 
                                SenseOfResponsibility, Sociability, 
                                created_at, updated_at
                         FROM skills 
                         $whereClause 
                         ORDER BY name, registration_no",
                        $params, 
                        "rows"
                    );
                } elseif (!empty($error_messages)) {
                    $error_message = implode("<br>", $error_messages);
                }
            }

            // Handle form submissions for deleting a record
            if (isset($_POST['delete']) && isset($_POST['delete_id'])) {
                $delete_id = strip_tags($_POST['delete_id']);
                
                // Get student name for confirmation message
                $student_info = sql($DBH, 
                    "SELECT name FROM skills WHERE id = ?", 
                    [$delete_id], 
                    "row"
                );
                
                $delete_query = "DELETE FROM skills WHERE id = ?";
                $stmt = $DBH->prepare($delete_query);
                
                if ($stmt->execute([$delete_id])) {
                    $success_message = "Successfully deleted skills record for: " . ($student_info['name'] ?? 'Unknown Student');
                    
                    // Refresh the results
                    $skills_results = sql($DBH, 
                        "SELECT id, name, registration_no, class, subclass, year, term, teacher_no,
                                Creativity, Fluency, Honesty, Leadership, 
                                Neatness, Politeness, Punctuality, SelfControl, 
                                SenseOfResponsibility, Sociability, 
                                created_at, updated_at
                         FROM skills 
                         $whereClause 
                         ORDER BY name, registration_no",
                        $params, 
                        "rows"
                    );
                    $total_results_count = count($skills_results);
                } else {
                    $error_message = "Failed to delete record";
                }
            }
        }
    }
} catch (PDOException $e) {
    $error_message = "Database Error: " . $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="en">

<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
  <title>VERBUM DEI ACADEMY - Psychomotor Skills Management</title>
  <link rel="icon" type="image/png" href="https://portal.vda.com.ng/pri_admin/pri_logo.jpg">
  
  <!-- Bootstrap 5 CSS -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
  
  <!-- Font Awesome -->
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
  
  <!-- Google Fonts - Nunito -->
  <link href="https://fonts.googleapis.com/css2?family=Nunito:wght@400;600;700&display=swap" rel="stylesheet">
  
  <!-- Custom CSS -->
  <style>
    :root {
      --primary-color: #1976d2;
      --secondary-color: #64b5f6;
      --accent-color: #bbdefb;
      --dark-color: #0d47a1;
      --light-color: #e3f2fd;
      --text-dark: #212529;
      --text-light: #f8f9fa;
      --skills-color: #4caf50;
      --affective-color: #ff9800;
    }
    
    body {
      font-family: 'Nunito', sans-serif;
      background-color: #f5f7fa;
      padding-top: 56px;
      color: var(--text-dark);
      font-size: 0.9rem;
    }
    
    /* Sidebar Styles */
    .sidebar {
      background: linear-gradient(180deg, var(--dark-color) 0%, var(--primary-color) 100%);
      position: fixed;
      top: 56px;
      left: 0;
      width: 250px;
      height: calc(100vh - 56px);
      transition: all 0.3s;
      z-index: 1000;
      overflow-y: auto;
    }
    
    .sidebar .nav-link {
      color: rgba(255, 255, 255, 0.85);
      font-weight: 600;
      padding: 0.75rem 1rem;
      margin-bottom: 0.2rem;
      transition: all 0.2s;
    }
    
    .sidebar .nav-link:hover {
      color: #fff;
      background-color: rgba(255, 255, 255, 0.15);
    }
    
    .sidebar .nav-link.active {
      color: #fff;
      background-color: var(--secondary-color);
      box-shadow: 0 2px 5px rgba(0,0,0,0.1);
    }
    
    .sidebar .nav-link i {
      margin-right: 0.5rem;
      width: 20px;
      text-align: center;
    }
    
    /* Main Content Styles */
    .main-content {
      margin-left: 250px;
      width: calc(100% - 250px);
      transition: all 0.3s;
      min-height: calc(100vh - 56px);
      padding-bottom: 60px;
    }
    
    @media (max-width: 991.98px) {
      .sidebar {
        left: -250px;
      }
      .main-content {
        margin-left: 0;
        width: 100%;
      }
      body {
        padding-top: 56px;
      }
    }
    
    /* Navbar */
    .navbar {
      background-color: var(--primary-color);
      box-shadow: 0 0.15rem 1.75rem 0 rgba(33, 37, 41, 0.1);
      position: fixed;
      top: 0;
      width: 100%;
      z-index: 1030;
      color: white;
    }
    
    /* Welcome banner */
    .welcome-banner {
      background: linear-gradient(135deg, var(--skills-color) 0%, #81c784 100%);
      color: white;
      border-radius: 0.5rem;
      padding: 0.75rem 1rem;
      margin: 1rem;
      box-shadow: 0 4px 6px rgba(0,0,0,0.1);
    }
    
    /* Form Styles */
    .form-container {
      background-color: white;
      border-radius: 0.5rem;
      padding: 1rem;
      margin: 1rem;
      box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.1);
    }
    
    /* Table Styles */
    .table-container {
      background-color: white;
      border-radius: 0.5rem;
      padding: 1rem;
      margin: 1rem;
      box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.1);
      overflow-x: auto;
    }
    
    .skills-table {
      width: 100%;
      border-collapse: collapse;
      font-size: 0.85rem;
    }
    
    .skills-table th,
    .skills-table td {
      padding: 0.5rem;
      border: 1px solid #dee2e6;
      text-align: center;
      vertical-align: middle;
    }
    
    .skills-table th {
      background-color: var(--skills-color);
      color: white;
      font-weight: 600;
      position: sticky;
      top: 0;
      z-index: 10;
    }
    
    .skills-table input,
    .skills-table select {
      width: 100%;
      padding: 0.25rem 0.5rem;
      border: 1px solid #ced4da;
      border-radius: 0.25rem;
      font-size: 0.85rem;
      text-align: center;
    }
    
    .skills-table input:focus,
    .skills-table select:focus {
      border-color: var(--skills-color);
      box-shadow: 0 0 0 0.2rem rgba(76, 175, 80, 0.25);
      outline: none;
    }
    
    .skills-table .student-name {
      text-align: left;
      font-weight: 600;
      background-color: #e8f5e9;
      position: sticky;
      left: 0;
      z-index: 5;
      min-width: 120px;
    }
    
    .skills-table .skill-col {
      background-color: #f1f8e9;
      min-width: 70px;
    }
    
    .skills-table .skill-col:hover {
      background-color: #e8f5e9;
    }
    
    .skills-table .registration-col {
      background-color: #e3f2fd;
      min-width: 100px;
    }
    
    .skills-table tr:nth-child(even) {
      background-color: #f9f9f9;
    }
    
    .skills-table tr:hover {
      background-color: #f1f1f1;
    }
    
    /* Footer styles */
    .footer {
      background-color: var(--dark-color);
      color: white;
      padding: 12px 0;
      position: fixed;
      bottom: 0;
      width: 100%;
      z-index: 100;
      font-size: 0.8rem;
    }
    
    .copyright-text {
      font-weight: 600;
    }
    
    .powered-by {
      display: flex;
      align-items: center;
      justify-content: flex-end;
      color: white;
    }
    
    .powered-by img {
      height: 18px;
      margin-left: 6px;
    }
    
    .powered-by-link {
      display: flex;
      align-items: center;
      text-decoration: none;
      color: inherit;
      transition: opacity 0.3s ease;
    }
    
    .powered-by-link:hover {
      opacity: 0.8;
      text-decoration: none;
    }
    
    /* Action buttons */
    .btn-action {
      padding: 0.5rem 1rem;
      border-radius: 0.25rem;
      font-weight: 600;
      transition: all 0.3s;
      font-size: 0.85rem;
    }
    
    .btn-primary {
      background-color: var(--primary-color);
      color: white;
      border: none;
    }
    
    .btn-primary:hover {
      background-color: var(--dark-color);
    }
    
    .btn-success {
      background-color: var(--skills-color);
      color: white;
      border: none;
    }
    
    .btn-success:hover {
      background-color: #388e3c;
    }
    
    .btn-danger {
      background-color: #dc3545;
      color: white;
      border: none;
    }
    
    .btn-danger:hover {
      background-color: #c82333;
    }
    
    /* Alert styles */
    .alert {
      padding: 0.75rem 1rem;
      border-radius: 0.25rem;
      margin: 1rem;
    }
    
    .alert-success {
      background-color: #d4edda;
      border-color: #c3e6cb;
      color: #155724;
    }
    
    .alert-danger {
      background-color: #f8d7da;
      border-color: #f5c6cb;
      color: #721c24;
    }
    
    /* Info cards */
    .info-card {
      background: linear-gradient(135deg, var(--light-color) 0%, #ffffff 100%);
      border-left: 4px solid var(--skills-color);
      padding: 0.75rem;
      margin-bottom: 0.5rem;
      border-radius: 0 0.25rem 0.25rem 0;
    }
    
    /* Compact form elements */
    .form-select, .form-label {
      font-size: 0.85rem;
    }
    
    /* Skills stats */
    .skills-stats {
      display: flex;
      gap: 1rem;
      margin-bottom: 1rem;
      flex-wrap: wrap;
    }
    
    .stat-card {
      background: linear-gradient(135deg, #e8f5e9 0%, #f5f5f5 100%);
      padding: 0.5rem 1rem;
      border-radius: 0.25rem;
      text-align: center;
      flex: 1;
      min-width: 120px;
    }
    
    .stat-value {
      font-size: 1.5rem;
      font-weight: bold;
      color: var(--skills-color);
    }
    
    .stat-label {
      font-size: 0.75rem;
      color: #666;
    }
    
    /* Skill rating colors */
    .rating-5 { background-color: #4caf50; color: white; }
    .rating-4 { background-color: #8bc34a; color: white; }
    .rating-3 { background-color: #cddc39; color: #333; }
    .rating-2 { background-color: #ffeb3b; color: #333; }
    .rating-1 { background-color: #ff9800; color: white; }
    
    select option.rating-5 { background-color: #4caf50; color: white; }
    select option.rating-4 { background-color: #8bc34a; color: white; }
    select option.rating-3 { background-color: #cddc39; color: #333; }
    select option.rating-2 { background-color: #ffeb3b; color: #333; }
    select option.rating-1 { background-color: #ff9800; color: white; }
    
    /* Rating guide */
    .rating-guide {
      display: flex;
      justify-content: space-around;
      margin-top: 10px;
      padding: 5px;
      background: #f5f5f5;
      border-radius: 5px;
    }
    
    .rating-item {
      text-align: center;
      padding: 5px;
    }
    
    .rating-circle {
      width: 25px;
      height: 25px;
      border-radius: 50%;
      display: flex;
      align-items: center;
      justify-content: center;
      margin: 0 auto 5px;
      font-size: 0.8rem;
      font-weight: bold;
    }
  </style>
</head>

<body>
  <!-- Top Navigation -->
  <nav class="navbar navbar-expand-lg navbar-light">
    <div class="container-fluid">
      <button id="sidebarToggle" class="btn btn-link d-lg-none me-2">
        <i class="fas fa-bars"></i>
      </button>
      <a class="navbar-brand d-none d-lg-block" href="#">
        VERBUM DEI ACADEMY
      </a>
      <div class="ms-auto d-flex align-items-center">
        <div class="admin-info me-3">
          <span class="admin-name"><?php echo htmlspecialchars($adminName ?? 'Admin'); ?></span>
          <span class="admin-access">(Administrator)</span>
          <i class="fas fa-user-circle admin-icon"></i>
        </div>
        <a href="logout.php" class="btn btn-sm btn-outline-light">
          <i class="fas fa-sign-out-alt"></i> Logout
        </a>
      </div>
    </div>
  </nav>

  <!-- Sidebar -->
  <nav id="sidebar" class="sidebar">
    <div class="position-sticky pt-3">
      <div class="text-center mb-4 d-none d-lg-block">
        <img src="https://portal.vda.com.ng/pri_admin/pri_logo.jpg" alt="School Logo" class="rounded-circle" width="80">
        <h5 class="mt-2 text-white">VERBUM DEI ACADEMY</h5>
        <p class="text-white-50 small">Admin Dashboard</p>
      </div>
      
      <ul class="nav flex-column">
        <li class="nav-item">
          <a class="nav-link" href="main.php">
            <i class="fas fa-fw fa-tachometer-alt"></i>
            Admin Dashboard
          </a>
        </li>
        <li class="nav-item">
          <a class="nav-link" href="edit_scores_subject.php">
            <i class="fas fa-fw fa-edit"></i>
            Edit Subject Scores
          </a>
        </li>
        <li class="nav-item">
          <a class="nav-link" href="download_broadsheet.php">
            <i class="fas fa-fw fa-file-pdf"></i>
            Result Broadsheet
          </a>
        </li>
        <li class="nav-item">
          <a class="nav-link" href="list_uploaded_results.php">
            <i class="fas fa-fw fa-list"></i>
            Uploaded Results
          </a>
        </li>
        <li class="nav-item">
          <a class="nav-link" href="edit_comments_attendance.php">
            <i class="fas fa-fw fa-comments"></i>
            Attendance & Comments
          </a>
        </li>
        <li class="nav-item">
          <a class="nav-link active" href="edit_skills.php">
            <i class="fas fa-fw fa-chart-bar"></i>
            Psychomotor Skills
          </a>
        </li>
      </ul>
    </div>
  </nav>

  <!-- Main Content -->
  <main class="main-content">
    <!-- Welcome Banner -->
    <div class="welcome-banner">
      <div class="d-flex justify-content-between align-items-center">
        <div>
          <h4 class="mb-1">Psychomotor Skills Management</h4>
          <p class="mb-0 small">VERBUM DEI ACADEMY Admin Dashboard</p>
        </div>
        <div class="d-none d-sm-inline-block">
          <span class="badge bg-white text-success p-1 small">
            <i class="fas fa-brain me-1"></i> Skills Assessment System
          </span>
        </div>
      </div>
    </div>

    <!-- Search Form -->
    <div class="form-container">
      <form method="POST" action="">
        <div class="row align-items-end">
          <div class="col-md-3">
            <div class="form-group mb-2">
              <label for="class" class="form-label">Class:</label>
              <select id="class" name="class" class="form-select form-select-sm">
                <option value="">All Classes</option>
                <?php foreach ($classes as $classOption): ?>
                  <option value="<?php echo htmlspecialchars($classOption['class']); ?>" 
                    <?php echo ($classOption['class'] === $class) ? 'selected' : ''; ?>>
                    <?php echo htmlspecialchars($classOption['class']); ?>
                  </option>
                <?php endforeach; ?>
              </select>
            </div>
          </div>
          
          <div class="col-md-2">
            <div class="form-group mb-2">
              <label for="term" class="form-label">Term:</label>
              <select id="term" name="term" class="form-select form-select-sm">
                <option value="">All Terms</option>
                <?php foreach ($terms as $termOption): ?>
                  <option value="<?php echo htmlspecialchars($termOption['term']); ?>" 
                    <?php echo ($termOption['term'] === $term) ? 'selected' : ''; ?>>
                    <?php echo htmlspecialchars($termOption['term']); ?>
                  </option>
                <?php endforeach; ?>
              </select>
            </div>
          </div>
          
          <div class="col-md-2">
            <div class="form-group mb-2">
              <label for="year" class="form-label">Academic Year:</label>
              <select id="year" name="year" class="form-select form-select-sm">
                <option value="">All Years</option>
                <?php foreach ($years as $yearOption): ?>
                  <option value="<?php echo htmlspecialchars($yearOption['year']); ?>" 
                    <?php echo ($yearOption['year'] === $year) ? 'selected' : ''; ?>>
                    <?php echo htmlspecialchars($yearOption['year']); ?>
                  </option>
                <?php endforeach; ?>
              </select>
            </div>
          </div>
          
          <div class="col-md-3">
            <div class="form-group mb-2">
              <label for="teacher_no" class="form-label">Teacher:</label>
              <select id="teacher_no" name="teacher_no" class="form-select form-select-sm">
                <option value="">All Teachers</option>
                <?php foreach ($teachers as $teacherOption): ?>
                  <option value="<?php echo htmlspecialchars($teacherOption['teacher_no']); ?>" 
                    <?php echo ($teacherOption['teacher_no'] === $teacher_no) ? 'selected' : ''; ?>>
                    <?php echo htmlspecialchars($teacherOption['teacher_no']); ?>
                  </option>
                <?php endforeach; ?>
              </select>
            </div>
          </div>
          
          <div class="col-md-2">
            <button type="submit" class="btn btn-success btn-sm w-100">
              <i class="fas fa-search me-1"></i> Search Records
            </button>
          </div>
        </div>
      </form>
    </div>

    <?php if (isset($error_message)): ?>
      <div class="alert alert-danger">
        <i class="fas fa-exclamation-circle me-2"></i> <?php echo $error_message; ?>
      </div>
    <?php endif; ?>

    <?php if (isset($success_message)): ?>
      <div class="alert alert-success">
        <i class="fas fa-check-circle me-2"></i> <?php echo $success_message; ?>
      </div>
    <?php endif; ?>

    <?php if (!empty($page_title) && (!empty($class) || !empty($term) || !empty($year) || !empty($teacher_no))): ?>
      <div class="form-container">
        <div class="row">
          <div class="col-md-8">
            <div class="info-card">
              <h6 class="mb-1"><i class="fas fa-edit me-2"></i>Modifying Psychomotor Skills For:</h6>
              <p class="mb-0">
                <?php 
                $filters = [];
                if (!empty($class)) $filters[] = "Class: " . htmlspecialchars($class);
                if (!empty($term)) $filters[] = "Term: " . htmlspecialchars($term);
                if (!empty($year)) $filters[] = "Year: " . htmlspecialchars($year);
                if (!empty($teacher_no)) $filters[] = "Teacher: " . htmlspecialchars($teacher_no);
                echo implode(", ", $filters);
                ?>
              </p>
            </div>
          </div>
          <div class="col-md-4">
            <div class="info-card">
              <h6 class="mb-1"><i class="fas fa-users me-2"></i>Total Records: <?php echo $total_results_count; ?></h6>
              <p class="mb-0 small">Number of skills records found</p>
            </div>
          </div>
        </div>
        
        <?php if (!empty($skills_results)): 
          // Calculate skills statistics
          $average_ratings = [
            'Creativity' => 0,
            'Fluency' => 0,
            'Honesty' => 0,
            'Leadership' => 0,
            'Neatness' => 0,
            'Politeness' => 0,
            'Punctuality' => 0,
            'SelfControl' => 0,
            'SenseOfResponsibility' => 0,
            'Sociability' => 0
          ];
          
          foreach ($skills_results as $record) {
            foreach ($average_ratings as $skill => $value) {
              $average_ratings[$skill] += intval($record[$skill]);
            }
          }
          
          foreach ($average_ratings as $skill => $value) {
            $average_ratings[$skill] = $total_results_count > 0 ? round($value / $total_results_count, 1) : 0;
          }
        ?>
          <div class="skills-stats">
            <?php foreach ($average_ratings as $skill => $avg): ?>
              <div class="stat-card">
                <div class="stat-value"><?php echo $avg; ?></div>
                <div class="stat-label"><?php echo htmlspecialchars($skill); ?></div>
              </div>
            <?php endforeach; ?>
          </div>
          
          <!-- Rating Guide -->
          <div class="rating-guide">
            <div class="rating-item">
              <div class="rating-circle rating-5">5</div>
              <div>Excellent</div>
            </div>
            <div class="rating-item">
              <div class="rating-circle rating-4">4</div>
              <div>Very Good</div>
            </div>
            <div class="rating-item">
              <div class="rating-circle rating-3">3</div>
              <div>Good</div>
            </div>
            <div class="rating-item">
              <div class="rating-circle rating-2">2</div>
              <div>Fair</div>
            </div>
            <div class="rating-item">
              <div class="rating-circle rating-1">1</div>
              <div>Poor</div>
            </div>
          </div>
        <?php endif; ?>
      </div>
    <?php endif; ?>

    <?php if (!empty($skills_results)): ?>
      <div class="table-container">
        <form method="POST">
          <input type="hidden" name="class" value="<?php echo htmlspecialchars($class); ?>">
          <input type="hidden" name="term" value="<?php echo htmlspecialchars($term); ?>">
          <input type="hidden" name="year" value="<?php echo htmlspecialchars($year); ?>">
          <input type="hidden" name="teacher_no" value="<?php echo htmlspecialchars($teacher_no); ?>">
          
          <div class="d-flex justify-content-between align-items-center mb-3">
            <h5 class="mb-0">
              Student Psychomotor Skills Assessment
              <small class="text-muted">(<?php echo $total_results_count; ?> records)</small>
            </h5>
            <div>
              <button type="submit" name="save_all" class="btn btn-success btn-sm">
                <i class="fas fa-save me-1"></i> Save All Changes
              </button>
              <a href="main.php" class="btn btn-primary btn-sm ms-2">
                <i class="fas fa-arrow-left me-1"></i> Back to Dashboard
              </a>
            </div>
          </div>
          
          <div class="table-responsive">
            <table class="skills-table">
              <thead>
                <tr>
                  <th width="4%">S/N</th>
                  <th class="student-name">Student Name</th>
                  <th class="registration-col">Reg No</th>
                  <th class="skill-col">Class</th>
                  <th class="skill-col">Term</th>
                  <th class="skill-col">Year</th>
                  <th class="skill-col" title="Creativity">Creat</th>
                  <th class="skill-col" title="Fluency">Fluen</th>
                  <th class="skill-col" title="Honesty">Honest</th>
                  <th class="skill-col" title="Leadership">Leader</th>
                  <th class="skill-col" title="Neatness">Neat</th>
                  <th class="skill-col" title="Politeness">Polite</th>
                  <th class="skill-col" title="Punctuality">Punct</th>
                  <th class="skill-col" title="Self Control">Self Ctrl</th>
                  <th class="skill-col" title="Sense of Responsibility">Respons</th>
                  <th class="skill-col" title="Sociability">Social</th>
                  <th width="8%">Actions</th>
                </tr>
              </thead>
              <tbody>
                <?php foreach ($skills_results as $index => $record): ?>
                  <tr>
                    <td><?php echo $index + 1; ?></td>
                    <td class="student-name"><?php echo htmlspecialchars($record['name']); ?></td>
                    <td class="registration-col"><?php echo htmlspecialchars($record['registration_no']); ?></td>
                    <td class="skill-col"><?php echo htmlspecialchars($record['class']); ?></td>
                    <td class="skill-col"><?php echo htmlspecialchars($record['term']); ?></td>
                    <td class="skill-col"><?php echo htmlspecialchars($record['year']); ?></td>
                    
                    <!-- Skills ratings as dropdowns -->
                    <?php 
                    $skills_fields = [
                        'Creativity', 'Fluency', 'Honesty', 'Leadership',
                        'Neatness', 'Politeness', 'Punctuality', 'SelfControl',
                        'SenseOfResponsibility', 'Sociability'
                    ];
                    
                    foreach ($skills_fields as $field): 
                        $value = htmlspecialchars($record[$field]);
                        $rating_class = "rating-" . $value;
                    ?>
                      <td class="skill-col">
                        <select name="<?php echo strtolower($field); ?>[]" class="form-select form-select-sm <?php echo $rating_class; ?>" 
                                onchange="this.className='form-select form-select-sm rating-'+this.value">
                          <option value="1" class="rating-1" <?php echo $value == '1' ? 'selected' : ''; ?>>1</option>
                          <option value="2" class="rating-2" <?php echo $value == '2' ? 'selected' : ''; ?>>2</option>
                          <option value="3" class="rating-3" <?php echo $value == '3' ? 'selected' : ''; ?>>3</option>
                          <option value="4" class="rating-4" <?php echo $value == '4' ? 'selected' : ''; ?>>4</option>
                          <option value="5" class="rating-5" <?php echo $value == '5' ? 'selected' : ''; ?>>5</option>
                        </select>
                      </td>
                    <?php endforeach; ?>
                    
                    <td>
                      <input type="hidden" name="id[]" value="<?php echo $record['id']; ?>">
                      <div class="btn-group btn-group-sm" role="group">
                        <button type="button" class="btn btn-danger" 
                                onclick="confirmDelete(<?php echo $record['id']; ?>, '<?php echo addslashes($record['name']); ?>')"
                                title="Delete Record">
                          <i class="fas fa-trash"></i>
                        </button>
                      </div>
                    </td>
                  </tr>
                <?php endforeach; ?>
              </tbody>
            </table>
          </div>
          
          <div class="mt-3">
            <button type="submit" name="save_all" class="btn btn-success btn-action">
              <i class="fas fa-save me-1"></i> Save All Changes
            </button>
            <button type="button" class="btn btn-secondary btn-action ms-2" onclick="resetForm()">
              <i class="fas fa-undo me-1"></i> Reset Changes
            </button>
          </div>
          
          <!-- Hidden delete form -->
          <input type="hidden" name="delete_id" id="delete_id" value="">
        </form>
        
        <div class="mt-3">
          <div class="alert alert-info py-2">
            <h6 class="mb-1"><i class="fas fa-info-circle me-1"></i> Instructions</h6>
            <p class="mb-0 small">
              1. Use dropdowns to edit skill ratings (1-5 scale).<br>
              2. 5=Excellent, 4=Very Good, 3=Good, 2=Fair, 1=Poor<br>
              3. All ratings must be between 1 and 5.<br>
              4. Click "Save All Changes" to update all records at once.<br>
              5. Use the delete button to remove individual records.<br>
              6. Use filters to narrow down records by class, term, year, or teacher.
            </p>
          </div>
        </div>
      </div>
    <?php elseif ($_SERVER["REQUEST_METHOD"] == "POST" && empty($skills_results)): ?>
      <div class="form-container text-center py-4">
        <i class="fas fa-search fa-3x mb-2 text-muted"></i>
        <h5 class="text-muted">No Records Found</h5>
        <p class="mb-0">There are no skills records for the selected criteria.</p>
      </div>
    <?php else: ?>
      <div class="form-container text-center py-4">
        <i class="fas fa-chart-bar fa-3x mb-2 text-muted"></i>
        <h5 class="text-muted">Psychomotor Skills Management</h5>
        <p class="mb-0">Select filters to search for and edit psychomotor skills records.</p>
        <p class="small text-muted mt-2">You can filter by class, term, year, or teacher. Leave filters empty to see all records.</p>
      </div>
    <?php endif; ?>
  </main>

  <!-- Footer -->
  <footer class="footer">
    <div class="container-fluid">
      <div class="row align-items-center">
        <div class="col-md-6">
          <span class="copyright-text">Copyright &copy; VERBUM DEI ACADEMY <?php echo date("Y"); ?></span>
        </div>
        <div class="col-md-6">
          <div class="powered-by">
            <a href="https://dataswift.com.ng" target="_blank" rel="noopener noreferrer" class="powered-by-link">
              <span>Powered by DataSwift</span>
              <img src="datalogo.png" alt="DataSwift Logo">
            </a>
          </div>
        </div>
      </div>
    </div>
  </footer>

  <!-- Bootstrap 5 JS Bundle with Popper -->
  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
  
  <!-- Custom Scripts -->
  <script>
    // Toggle sidebar on mobile
    document.getElementById('sidebarToggle').addEventListener('click', function() {
      document.getElementById('sidebar').classList.toggle('active');
      document.querySelector('.main-content').classList.toggle('active');
    });
    
    // Close sidebar when clicking outside on mobile
    document.addEventListener('click', function(event) {
      const sidebar = document.getElementById('sidebar');
      const sidebarToggle = document.getElementById('sidebarToggle');
      
      if (window.innerWidth < 992 && 
          !sidebar.contains(event.target) && 
          !sidebarToggle.contains(event.target) && 
          sidebar.classList.contains('active')) {
        sidebar.classList.remove('active');
        document.querySelector('.main-content').classList.remove('active');
      }
    });
    
    // Delete confirmation function
    function confirmDelete(id, name) {
      if (confirm(`Are you sure you want to delete the skills record for:\n\n${name}?\n\nThis action cannot be undone!`)) {
        document.getElementById('delete_id').value = id;
        // Create a form to submit the delete request
        const form = document.createElement('form');
        form.method = 'POST';
        form.style.display = 'none';
        
        // Add hidden inputs
        const classInput = document.createElement('input');
        classInput.type = 'hidden';
        classInput.name = 'class';
        classInput.value = '<?php echo htmlspecialchars($class); ?>';
        form.appendChild(classInput);
        
        const termInput = document.createElement('input');
        termInput.type = 'hidden';
        termInput.name = 'term';
        termInput.value = '<?php echo htmlspecialchars($term); ?>';
        form.appendChild(termInput);
        
        const yearInput = document.createElement('input');
        yearInput.type = 'hidden';
        yearInput.name = 'year';
        yearInput.value = '<?php echo htmlspecialchars($year); ?>';
        form.appendChild(yearInput);
        
        const teacherInput = document.createElement('input');
        teacherInput.type = 'hidden';
        teacherInput.name = 'teacher_no';
        teacherInput.value = '<?php echo htmlspecialchars($teacher_no); ?>';
        form.appendChild(teacherInput);
        
        const deleteIdInput = document.createElement('input');
        deleteIdInput.type = 'hidden';
        deleteIdInput.name = 'delete_id';
        deleteIdInput.value = id;
        form.appendChild(deleteIdInput);
        
        const deleteInput = document.createElement('input');
        deleteInput.type = 'hidden';
        deleteInput.name = 'delete';
        deleteInput.value = '1';
        form.appendChild(deleteInput);
        
        document.body.appendChild(form);
        form.submit();
      }
    }
    
    // Reset form function
    function resetForm() {
      if (confirm('Are you sure you want to reset all changes?')) {
        location.reload();
      }
    }
    
    // Validate rating inputs on change
    document.querySelectorAll('select').forEach(select => {
      select.addEventListener('change', function() {
        const value = parseInt(this.value);
        if (value < 1 || value > 5) {
          alert('Rating must be between 1 and 5');
          this.value = '';
          this.focus();
        }
      });
    });
    
    // Focus first input field if records exist
    <?php if (!empty($skills_results)): ?>
      document.addEventListener('DOMContentLoaded', function() {
        const firstSelect = document.querySelector('select');
        if (firstSelect) {
          firstSelect.focus();
        }
      });
    <?php endif; ?>
    
    // Update dropdown colors based on selection
    document.addEventListener('change', function(e) {
      if (e.target.tagName === 'SELECT') {
        const value = e.target.value;
        // Remove all rating classes
        e.target.className = 'form-select form-select-sm';
        // Add appropriate rating class
        if (value >= 1 && value <= 5) {
          e.target.classList.add('rating-' + value);
        }
      }
    });
    
    // Initialize dropdown colors on page load
    document.addEventListener('DOMContentLoaded', function() {
      document.querySelectorAll('select').forEach(select => {
        const value = select.value;
        if (value >= 1 && value <= 5) {
          select.classList.add('rating-' + value);
        }
      });
    });
  </script>
</body>
</html>